package ru.yandex.solomon.auth.tvm;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;


/**
 * @author Sergey Polovko
 */
public abstract class TvmSubject implements AuthSubject {

    public static TvmSubject user(long uid, String login) {
        return new UserSubject(uid, login);
    }

    public static TvmSubject service(int clientId) {
        return new ServiceSubject(clientId);
    }

    /**
     * USER SUBJECT
     */
    public static final class UserSubject extends TvmSubject {

        private final long uid;
        private final String login;

        UserSubject(long uid, String login) {
            this.uid = uid;
            this.login = login;
        }

        @Override
        public String getUniqueId() {
            return login;
        }

        public long getUid() {
            return uid;
        }

        @Override
        public AuthType getAuthType() {
            return AuthType.TvmUser;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            UserSubject that = (UserSubject) o;
            return uid == that.uid;
        }

        @Override
        public int hashCode() {
            return (int) (uid ^ (uid >>> 32));
        }

        @Override
        public String toString() {
            return "TvmUserSubject{uid=" + uid + ", login='" + login + "'}";
        }
    }

    /**
     * SERVICE SUBJECT
     */
    public static final class ServiceSubject extends TvmSubject {
        private final int clientId;

        ServiceSubject(int clientId) {
            this.clientId = clientId;
        }

        @Override
        public String getUniqueId() {
            return FakeTvmLogin.forClientId(clientId);
        }

        public int getClientId() {
            return clientId;
        }

        @Override
        public AuthType getAuthType() {
            return AuthType.TvmService;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            ServiceSubject service = (ServiceSubject) o;
            return clientId == service.clientId;
        }

        @Override
        public int hashCode() {
            return clientId;
        }

        @Override
        public String toString() {
            return "TvmServiceSubject{clientId=" + clientId + '}';
        }
    }
}
