package ru.yandex.solomon.balancer;

import java.time.Clock;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.balancer.dao.BalancerDao;
import ru.yandex.solomon.balancer.remote.RemoteCluster;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockDetail;
import ru.yandex.solomon.locks.LockSubscriber;
import ru.yandex.solomon.locks.UnlockReason;

/**
 * @author Vladimir Gordiychuk
 */
public class BalancerHolderImpl implements BalancerHolder, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(BalancerHolderImpl.class);
    private static final long LEASE_EXPIRATION_MILLIS = TimeUnit.SECONDS.toMillis(15);

    private final Clock clock;
    private final DistributedLock lock;
    private final ExecutorService executor;
    private final ScheduledExecutorService timer;
    private final ShardsHolder shardsHolder;
    private final TotalShardCounter shardCounter;
    private final BalancerDao dao;
    private final List<Resource> resources;
    private final RemoteCluster cluster;
    private volatile boolean canceled;
    @Nullable
    private volatile Balancer state;

    public BalancerHolderImpl(
        Clock clock,
        DistributedLock lock,
        List<Resource> resources,
        RemoteCluster cluster,
        ExecutorService executor,
        ScheduledExecutorService timer,
        ShardsHolder shardsHolder,
        TotalShardCounter shardCounter,
        BalancerDao dao)
    {
        this.clock = clock;
        this.lock = lock;
        this.resources = resources;
        this.executor = executor;
        this.timer = timer;
        this.shardsHolder = shardsHolder;
        this.shardCounter = shardCounter;
        this.dao = dao;
        this.cluster = cluster;
        this.state = null;
    }

    private void tryAcquireLeadership() {
        if (canceled) {
            return;
        }

        lock.acquireLock(new LeaderSubscription(), LEASE_EXPIRATION_MILLIS, TimeUnit.MILLISECONDS);
    }

    public void start() {
        canceled = false;
        tryAcquireLeadership();
    }

    public void stop() {
        close();
    }

    private synchronized void becomeLeader(long seqNo) {
        var prev = this.state;
        this.state = new BalancerImpl(
            clock,
            seqNo,
            resources,
            cluster,
            timer,
            executor,
            shardsHolder,
            shardCounter,
            dao);
        close(prev);
    }

    private synchronized void becomeMember() {
        var prev = this.state;
        this.state = null;
        close(prev);
    }

    @Nullable
    public synchronized Balancer getState() {
        return state;
    }

    @Override
    public void close() {
        canceled = true;
        close(state);
        try {
            // do not join on future forever to prevent process hangup
            lock.unlock().get(3, TimeUnit.SECONDS);
        } catch (TimeoutException e) {
            logger.warn("cannot unlock {} in 3 seconds", lock);
        } catch (Exception e) {
            throw new RuntimeException("cannot unlock " + lock);
        }
    }

    private void close(@Nullable Balancer state) {
        if (state != null) {
            state.close();
        }
    }

    @Nullable
    @Override
    public String getLeaderNode() {
        return lock.lockDetail().map(LockDetail::owner).orElse(null);
    }

    @Nullable
    @Override
    public Balancer getBalancer() {
        return state;
    }

    private class LeaderSubscription implements LockSubscriber {
        @Override
        public boolean isCanceled() {
            return canceled;
        }

        @Override
        public void onLock(long seqNo) {
            logger.info("Become balancer, leader seqNo {}", seqNo);
            becomeLeader(seqNo);
        }

        @Override
        public void onUnlock(UnlockReason reason) {
            logger.info("Not balancer anymore: {}", reason);
            becomeMember();
            tryAcquireLeadership();
        }
    }
}
