package ru.yandex.solomon.balancer;

import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import com.google.common.primitives.Doubles;

import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Vladimir Gordiychuk
 */
public class BalancerOptions extends DefaultObject {
    private final long version;
    private final long heartbeatExpirationMillis;
    private final long forceUnassignExpirationMillis;
    private final long gracefulUnassignExpirationMillis;
    private final long assignExpirationMillis;
    private final int maxReassignInFlight;
    private final int maxLongLoadingShardsToIgnore;
    private final boolean enableAutoRebalance;
    private final double autoRebalanceDispersionThreshold;
    private final double rebalanceDispersionTarget;
    private final Resources limits;
    private final boolean disableAutoFreeze;

    public BalancerOptions(Builder builder) {
        this.version = builder.version;
        this.heartbeatExpirationMillis = builder.heartbeatExpirationMillis <= 0
            ? TimeUnit.MINUTES.toMillis(2)
            : builder.heartbeatExpirationMillis;

        this.forceUnassignExpirationMillis = builder.forceUnassignExpirationMillis <= 0
            ? TimeUnit.SECONDS.toMillis(10)
            : builder.forceUnassignExpirationMillis;

        this.gracefulUnassignExpirationMillis = builder.gracefulUnassignExpirationMillis <= 0
            ? TimeUnit.MINUTES.toMillis(1)
            : builder.gracefulUnassignExpirationMillis;

        this.assignExpirationMillis = builder.assignExpirationMillis <= 0
            ? TimeUnit.SECONDS.toMillis(15)
            : builder.assignExpirationMillis;

        this.maxReassignInFlight = builder.maxReassignInFlight <= 0 ? 50 : builder.maxReassignInFlight;
        this.maxLongLoadingShardsToIgnore = builder.maxLongLoadingShardsToIgnore;

        this.enableAutoRebalance = builder.enableAutoRebalance;

        var threshold = builder.autoRebalanceDispersionThreshold;
        this.autoRebalanceDispersionThreshold = Double.compare(threshold, 0.) <= 0 || Double.compare(threshold, 1.) > 0
            ? 1.
            : threshold;

        var target = builder.rebalanceDispersionTarget;
        this.rebalanceDispersionTarget = Doubles.constrainToRange(target, 0., 1.);

        this.limits = builder.limits == null || Resources.EMPTY.equals(builder.limits)
            ? new Resources()
            : builder.limits;

        this.disableAutoFreeze = builder.disableAutoFreeze;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public long getVersion() {
        return version;
    }

    public long getHeartbeatExpirationMillis() {
        return heartbeatExpirationMillis;
    }

    public long getForceUnassignExpirationMillis() {
        return forceUnassignExpirationMillis;
    }

    public long getGracefulUnassignExpirationMillis() {
        return gracefulUnassignExpirationMillis;
    }

    public long getAssignExpirationMillis() {
        return assignExpirationMillis;
    }

    public int getMaxReassignInFlight() {
        return maxReassignInFlight;
    }

    public int getMaxLongLoadingShardsToIgnore() {
        return maxLongLoadingShardsToIgnore;
    }

    public boolean isEnableAutoRebalance() {
        return enableAutoRebalance;
    }

    public double getAutoRebalanceDispersionThreshold() {
        return autoRebalanceDispersionThreshold;
    }

    public double getRebalanceDispersionTarget() {
        return rebalanceDispersionTarget;
    }

    public Resources getLimits() {
        return limits;
    }

    public boolean isDisableAutoFreeze() {
        return disableAutoFreeze;
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    @Override
    public String toString() {
        return "BalancerOptions{" +
            "version=" + version +
            ", heartbeatExpirationMillis=" + heartbeatExpirationMillis +
            ", forceUnassignExpirationMillis=" + forceUnassignExpirationMillis +
            ", gracefulUnassignExpirationMillis=" + gracefulUnassignExpirationMillis +
            ", assignExpirationMillis=" + assignExpirationMillis +
            ", maxReassignInFlight=" + maxReassignInFlight +
            ", maxLongLoadingShardsToIgnore=" + maxLongLoadingShardsToIgnore +
            ", enableAutoRebalance=" + enableAutoRebalance +
            ", autoRebalanceDispersionThreshold=" + autoRebalanceDispersionThreshold +
            ", rebalanceDispersionTarget=" + rebalanceDispersionTarget +
            ", limits=" + limits +
            ", disableAutoFreeze=" + disableAutoFreeze +
            '}';
    }

    public static final class Builder {
        private long version;
        private long heartbeatExpirationMillis;
        private long forceUnassignExpirationMillis;
        private long gracefulUnassignExpirationMillis;
        private long assignExpirationMillis;
        private int maxReassignInFlight;
        private int maxLongLoadingShardsToIgnore;
        private boolean enableAutoRebalance;
        private double autoRebalanceDispersionThreshold;
        private double rebalanceDispersionTarget;
        @Nullable
        private Resources limits;
        private boolean disableAutoFreeze;

        private Builder() {
        }

        private Builder(BalancerOptions opts) {
            this.version = opts.version;
            this.heartbeatExpirationMillis = opts.heartbeatExpirationMillis;
            this.forceUnassignExpirationMillis = opts.forceUnassignExpirationMillis;
            this.gracefulUnassignExpirationMillis = opts.gracefulUnassignExpirationMillis;
            this.assignExpirationMillis = opts.assignExpirationMillis;
            this.maxReassignInFlight = opts.maxReassignInFlight;
            this.maxLongLoadingShardsToIgnore = opts.maxLongLoadingShardsToIgnore;
            this.enableAutoRebalance = opts.enableAutoRebalance;
            this.autoRebalanceDispersionThreshold = opts.autoRebalanceDispersionThreshold;
            this.rebalanceDispersionTarget = opts.rebalanceDispersionTarget;
            this.limits = new Resources(opts.limits);
            this.disableAutoFreeze = opts.disableAutoFreeze;
        }

        public Builder setVersion(long version) {
            this.version = version;
            return this;
        }

        public Builder setHeartbeatExpiration(long time, TimeUnit unit) {
            this.heartbeatExpirationMillis = unit.toMillis(time);
            return this;
        }

        public Builder setForceUnassignExpiration(long time, TimeUnit unit) {
            this.forceUnassignExpirationMillis = unit.toMillis(time);
            return this;
        }

        public Builder setGracefulUnassignExpiration(long time, TimeUnit unit) {
            this.gracefulUnassignExpirationMillis = unit.toMillis(time);
            return this;
        }

        public Builder setAssignExpiration(long time, TimeUnit unit) {
            this.assignExpirationMillis = unit.toMillis(time);
            return this;
        }

        public Builder setMaxReassignInFlight(int max) {
            this.maxReassignInFlight = max;
            return this;
        }

        public Builder setMaxLongLoadingShardsToIgnore(int max) {
            this.maxLongLoadingShardsToIgnore = max;
            return this;
        }

        public Builder setEnableAutoRebalance(boolean enable) {
            this.enableAutoRebalance = enable;
            return this;
        }

        public Builder setAutoRebalanceDispersionThreshold(double threshold) {
            this.autoRebalanceDispersionThreshold = threshold;
            return this;
        }

        public Builder setRebalanceDispersionTarget(double target) {
            this.rebalanceDispersionTarget = target;
            return this;
        }

        public Builder setLimits(Resources limits) {
            this.limits = limits;
            return this;
        }

        public Builder setDisableAutoFreeze(boolean disable) {
            this.disableAutoFreeze = disable;
            return this;
        }

        public BalancerOptions build() {
            return new BalancerOptions(this);
        }
    }
}
