package ru.yandex.solomon.balancer;

import java.util.ArrayList;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import com.google.common.base.Strings;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2IntArrayMap;

import ru.yandex.solomon.balancer.TShardSummary.EStatus;
import ru.yandex.solomon.balancer.protobuf.TAssignmentSeqNo;
import ru.yandex.solomon.balancer.protobuf.TBalancerAssignmentState;
import ru.yandex.solomon.balancer.protobuf.TBalancerNode;
import ru.yandex.solomon.balancer.protobuf.TBalancerOptions;
import ru.yandex.solomon.balancer.protobuf.TBalancerResources;
import ru.yandex.solomon.balancer.protobuf.TBalancerShard;
import ru.yandex.solomon.balancer.remote.RemoteNodeState;
import ru.yandex.solomon.balancer.remote.RemoteShardState;
import ru.yandex.solomon.balancer.snapshot.SnapshotAssignments;
import ru.yandex.solomon.balancer.snapshot.SnapshotNode;
import ru.yandex.solomon.balancer.snapshot.SnapshotShard;

import static ru.yandex.solomon.balancer.CommonResource.ALERTS_COUNT;
import static ru.yandex.solomon.balancer.CommonResource.CPU;
import static ru.yandex.solomon.balancer.CommonResource.MEMORY;
import static ru.yandex.solomon.balancer.CommonResource.METRICS_COUNT;
import static ru.yandex.solomon.balancer.CommonResource.METRICS_PARSE_RATE;
import static ru.yandex.solomon.balancer.CommonResource.METRICS_READ_RATE;
import static ru.yandex.solomon.balancer.CommonResource.NETWORK;
import static ru.yandex.solomon.balancer.CommonResource.RECORDS_WRITE_RATE;
import static ru.yandex.solomon.balancer.CommonResource.RESOURCES_COUNT;
import static ru.yandex.solomon.balancer.CommonResource.SHARDS_COUNT;

/**
 * @author Vladimir Gordiychuk
 */
public class BalancerProto {
    public static TBalancerAssignmentState toProto(SnapshotAssignments assignments) {
        var nodeToId = new Object2IntArrayMap<String>(assignments.nodes.size());
        var state = TBalancerAssignmentState.newBuilder();
        for (var node : assignments.nodes) {
            int id = state.getNodesCount() + 1;
            state.addNodes(TBalancerNode.newBuilder()
                    .setId(id)
                    .setAddress(node.address)
                    .setActive(node.active)
                    .setFreeze(node.freeze)
                    .build());
            nodeToId.put(node.address, id);
        }

        for (var shard : assignments.shards) {
            var proto = TBalancerShard.newBuilder()
                    .setShardId(shard.shardId)
                    .setResources(toProto(shard.resources));

            if (shard.assignmentSeqNo != null) {
                proto.setAssignmentSeqNo(toProto(shard.assignmentSeqNo));
            }

            if (!Strings.isNullOrEmpty(shard.node)) {
                proto.setNodeId(nodeToId.getInt(shard.node));
            }

            state.addShards(proto.build());
        }

        return state.build();
    }

    public static SnapshotAssignments fromProto(TBalancerAssignmentState proto) {
        var nodeAddressById = new Int2ObjectOpenHashMap<String>();
        var nodes = new ArrayList<SnapshotNode>(proto.getNodesCount());
        for (var nodeProto : proto.getNodesList()) {
            var node = new SnapshotNode();
            node.active = nodeProto.getActive();
            node.freeze = nodeProto.getFreeze();
            node.address = nodeProto.getAddress();
            nodeAddressById.put(nodeProto.getId(), node.address);
            nodes.add(node);
        }

        var shards = new ArrayList<SnapshotShard>(proto.getShardsCount());
        for (var shardProto : proto.getShardsList()) {
            var shard = new SnapshotShard();
            shard.shardId = shardProto.getShardId();
            shard.node = nodeAddressById.get(shardProto.getNodeId());
            shard.assignmentSeqNo = fromProto(shardProto.getAssignmentSeqNo());
            shard.status = ShardStatus.UNKNOWN;
            shard.resources = fromProto(shardProto.getResources());
            shards.add(shard);
        }

        return new SnapshotAssignments(nodes, shards);
    }

    public static RemoteNodeState fromProto(TPingResponse response) {
        var state = new RemoteNodeState();
        state.address = response.getNode();
        {
            TNodeSummary proto = response.getNodeSummary();
            state.uptimeMillis = proto.getUpTimeMillis();
            state.memoryBytes = proto.getMemoryBytes();
            state.utimeMillis = proto.getUtimeMillis();
            state.receivedAt = System.currentTimeMillis();
        }

        {
            state.shards = new ArrayList<>(response.getShardSummaryCount());
            for (var proto : response.getShardSummaryList()) {
                var shard = new RemoteShardState();
                shard.shardId = proto.getShardId();
                shard.status = fromProto(proto.getStatus());
                shard.uptimeMillis = proto.getUpTimeMillis();
                shard.resources = fromProto(proto.getResources());
                state.shards.add(shard);
            }
        }

        return state;
    }

    public static ShardStatus fromProto(EStatus proto) {
        switch (proto) {
            case READY:
                return ShardStatus.READY;
            case LOADING:
                return ShardStatus.LOADING;
            case INIT:
                return ShardStatus.INIT;
            default:
                return ShardStatus.UNKNOWN;
        }
    }

    public static BalancerOptions fromProto(TBalancerOptions proto) {
        if (proto == null || TBalancerOptions.getDefaultInstance().equals(proto)) {
            return BalancerOptions.newBuilder().build();
        }

        return BalancerOptions.newBuilder()
                .setVersion(proto.getVersion())
                .setAssignExpiration(proto.getAssignExpirationMillis(), TimeUnit.MILLISECONDS)
                .setHeartbeatExpiration(proto.getHeartbeatExpirationMillis(), TimeUnit.MILLISECONDS)
                .setGracefulUnassignExpiration(proto.getGracefulUnassignExpirationMillis(), TimeUnit.MILLISECONDS)
                .setForceUnassignExpiration(proto.getForceUnassignExpirationMillis(), TimeUnit.MILLISECONDS)
                .setAutoRebalanceDispersionThreshold(proto.getAutoReassignDispersionThreshold())
                .setLimits(fromProto(proto.getLimits()))
                .setMaxReassignInFlight(Math.toIntExact(proto.getMaxReassignInFlight()))
                .setMaxLongLoadingShardsToIgnore(Math.toIntExact(proto.getMaxLongLoadingShardsToIgnore()))
                .setEnableAutoRebalance(proto.getEnableAutoRebalance())
                .setRebalanceDispersionTarget(proto.getRebalanceDispersionTarget())
                .setDisableAutoFreeze(proto.getDisableAutoFreeze())
                .build();
    }

    public static TBalancerOptions toProto(BalancerOptions opts) {
        return TBalancerOptions.newBuilder()
                .setVersion(opts.getVersion())
                .setAssignExpirationMillis(opts.getAssignExpirationMillis())
                .setHeartbeatExpirationMillis(opts.getHeartbeatExpirationMillis())
                .setGracefulUnassignExpirationMillis(opts.getGracefulUnassignExpirationMillis())
                .setForceUnassignExpirationMillis(opts.getForceUnassignExpirationMillis())
                .setAutoReassignDispersionThreshold(opts.getAutoRebalanceDispersionThreshold())
                .setLimits(toProto(opts.getLimits()))
                .setMaxReassignInFlight(opts.getMaxReassignInFlight())
                .setMaxLongLoadingShardsToIgnore(opts.getMaxLongLoadingShardsToIgnore())
                .setEnableAutoRebalance(opts.isEnableAutoRebalance())
                .setRebalanceDispersionTarget(opts.getRebalanceDispersionTarget())
                .setDisableAutoFreeze(opts.isDisableAutoFreeze())
                .build();
    }

    public static TBalancerResources toProto(Resources resources) {
        return TBalancerResources.newBuilder()
                .setCpu(resources.get(CPU))
                .setMemory(Math.round(resources.get(MEMORY)))
                .setShardsCount(Math.round(resources.get(SHARDS_COUNT)))
                .setAlertsCount(Math.round(resources.get(ALERTS_COUNT)))
                .setMetricsReadRate(resources.get(METRICS_READ_RATE))
                .setRecordsWriteRate(resources.get(RECORDS_WRITE_RATE))
                .setResourcesCount(Math.round(resources.get(RESOURCES_COUNT)))
                .setMetricsParseRate(resources.get(METRICS_PARSE_RATE))
                .setMetricsCount(Math.round(resources.get(METRICS_COUNT)))
                .setNetworkBytes(Math.round(resources.get(NETWORK)))
                .build();
    }

    public static Resources fromProto(TBalancerResources proto) {
        var result = new Resources();
        if (proto.equals(TBalancerResources.getDefaultInstance())) {
            return result;
        }

        setNotZero(result, CPU, proto.getCpu());
        setNotZero(result, MEMORY, proto.getMemory());
        setNotZero(result, NETWORK, proto.getNetworkBytes());
        setNotZero(result, SHARDS_COUNT, proto.getShardsCount());
        setNotZero(result, ALERTS_COUNT, proto.getAlertsCount());
        setNotZero(result, RESOURCES_COUNT, proto.getResourcesCount());
        setNotZero(result, METRICS_COUNT, proto.getMetricsCount());
        setNotZero(result, RECORDS_WRITE_RATE, proto.getRecordsWriteRate());
        setNotZero(result, METRICS_READ_RATE, proto.getMetricsReadRate());
        setNotZero(result, METRICS_PARSE_RATE, proto.getMetricsParseRate());
        return result;
    }

    static void setNotZero(Resources resources, Resource resource, double value) {
        if (value != 0.0) {
            resources.set(resource, value);
        }
    }

    public static TAssignmentSeqNo toProto(AssignmentSeqNo assignmentSeqNo) {
        if (assignmentSeqNo == null) {
            return TAssignmentSeqNo.getDefaultInstance();
        }

        return TAssignmentSeqNo.newBuilder()
                .setLeaderSeqNo(assignmentSeqNo.getLeaderSeqNo())
                .setAssignSeqNo(assignmentSeqNo.getAssignSeqNo())
                .build();
    }

    @Nullable
    public static AssignmentSeqNo fromProto(TAssignmentSeqNo proto) {
        if (proto.equals(TAssignmentSeqNo.getDefaultInstance())) {
            return null;
        }

        return new AssignmentSeqNo(proto.getLeaderSeqNo(), proto.getAssignSeqNo());
    }
}
