package ru.yandex.solomon.balancer;

import java.util.Arrays;
import java.util.Collection;

/**
 * @author Vladimir Gordiychuk
 */
public class ClusterState {

    public static int countUnknown(Collection<NodeSummary> nodes) {
        int count = 0;
        for (var node : nodes) {
            if (!node.isActive()) {
                continue;
            }

            if (node.getStatus() == NodeStatus.UNKNOWN) {
                count++;
            }
        }
        return count;
    }

    public static int countNearExpire(Collection<NodeSummary> nodes, long timeToExpire, long now) {
        int count = 0;
        for (var node : nodes) {
            if (!node.isActive()) {
                continue;
            }

            long expiredAfterMillis = node.getExpiredAt() - now;
            if (expiredAfterMillis <= timeToExpire) {
                count++;
            }
        }
        return count;
    }

    public static long uptimeMillis(Collection<NodeSummary> nodes) {
        long[] uptimes = new long[nodes.size()];
        int size = 0;
        for (var node : nodes) {
            if (node.getStatus() == NodeStatus.CONNECTED) {
                uptimes[size++] = node.getUptimeMillis();
            }
        }

        Arrays.sort(uptimes, 0, size);
        switch (size) {
            case 0:
                return 0;
            case 1:
                return uptimes[0];
            default:
                return uptimes[(size - 1) / 2];
        }
    }

    public static NodeSummary findBestNode(ShardSummary shard, Collection<NodeSummary> nodes, Resources maximum) {
        NodeSummary bestNode = null;
        double bestUsage = 0;

        if (shard.getNode() != null && shard.getNode().isActive()) {
            bestNode = shard.getNode();
            bestUsage = bestNode.getUsageWith(shard, maximum);
        }

        if (bestNode != null && bestNode.isFreeze()) {
            return bestNode;
        }

        for (NodeSummary node : nodes) {
            if (node.getStatus() != NodeStatus.CONNECTED) {
                continue;
            }

            if (!node.isActive() || node.isFreeze()) {
                continue;
            }

            if (Double.compare(node.getFailCommandPercent(), 0.5) >= 0) {
                continue;
            }

            double usage = node.getUsageWith(shard, maximum);
            if (bestNode == null || usage < bestUsage) {
                bestNode = node;
                bestUsage = usage;
            }
        }

        return bestNode;
    }
}
