package ru.yandex.solomon.balancer;

import java.util.Collection;
import java.util.concurrent.TimeUnit;

import ru.yandex.misc.dataSize.DataSize;

/**
 * @author Vladimir Gordiychuk
 */
public enum CommonResource implements Resource {
    CPU() {
        @Override
        public double maximum(Collection<ShardSummary> shards, Collection<NodeSummary> nodes) {
            return TimeUnit.SECONDS.toMillis(Runtime.getRuntime().availableProcessors());
        }

        @Override
        public String prettyName() {
            return "Cpu";
        }

        @Override
        public String prettyValue(double value) {
            return String.valueOf(Math.round(value));
        }
    },
    MEMORY() {
        @Override
        public double maximum(Collection<ShardSummary> shards, Collection<NodeSummary> nodes) {
            return Runtime.getRuntime().maxMemory();
        }

        @Override
        public String prettyName() {
            return "Memory";
        }

        @Override
        public String prettyValue(double value) {
            return DataSize.shortString(Math.round(value));
        }
    },
    NETWORK() {
        @Override
        public String prettyName() {
            return "Network/bytes";
        }

        @Override
        public String prettyValue(double value) {
            return DataSize.shortString(Math.round(value));
        }

        @Override
        public double maximum(Collection<ShardSummary> shards, Collection<NodeSummary> nodes) {
            return evenlyAcrossCluster(this, shards, nodes);
        }
    },

    SHARDS_COUNT() {
        @Override
        public double maximum(Collection<ShardSummary> shards, Collection<NodeSummary> nodes) {
            double halfCluster = nodes.stream().map(NodeSummary::isActive).count() * 0.5;
            if (halfCluster <= 1) {
                return shards.size();
            }
            return Math.ceil(shards.size() / halfCluster);
        }

        @Override
        public String prettyName() {
            return "Shards";
        }

        @Override
        public String prettyValue(double value) {
            return String.valueOf(Math.round(value));
        }
    },
    /**
     * Alerting resource
     */
    ALERTS_COUNT() {
        @Override
        public String prettyName() {
            return "Alerts";
        }

        @Override
        public String prettyValue(double value) {
            return DataSize.shortString(Math.round(value));
        }

        @Override
        public double maximum(Collection<ShardSummary> shards, Collection<NodeSummary> nodes) {
            return 2 * evenlyAcrossCluster(this, shards, nodes);
        }
    },
    /**
     * name-resolver resource
     */
    RESOURCES_COUNT() {
        @Override
        public String prettyName() {
            return "Resources";
        }

        @Override
        public String prettyValue(double value) {
            return DataSize.shortString(Math.round(value));
        }

        @Override
        public double maximum(Collection<ShardSummary> shards, Collection<NodeSummary> nodes) {
            return 2 * evenlyAcrossCluster(this, shards, nodes);
        }
    },
    METRICS_COUNT() {
        @Override
        public String prettyName() {
            return "Metrics";
        }

        @Override
        public String prettyValue(double value) {
            return DataSize.shortString(Math.round(value));
        }

        @Override
        public double maximum(Collection<ShardSummary> shards, Collection<NodeSummary> nodes) {
            return 2 * evenlyAcrossCluster(this, shards, nodes);
        }
    },

    RECORDS_WRITE_RATE() {
        @Override
        public String prettyName() {
            return "Writes/sec";
        }

        @Override
        public String prettyValue(double value) {
            return DataSize.shortString(Math.round(value));
        }

        @Override
        public double maximum(Collection<ShardSummary> shards, Collection<NodeSummary> nodes) {
            return evenlyAcrossCluster(this, shards, nodes);
        }
    },
    METRICS_READ_RATE() {
        @Override
        public String prettyName() {
            return "Reads/sec";
        }

        @Override
        public String prettyValue(double value) {
            return DataSize.shortString(Math.round(value));
        }

        @Override
        public double maximum(Collection<ShardSummary> shards, Collection<NodeSummary> nodes) {
            return evenlyAcrossCluster(this, shards, nodes);
        }
    },
    METRICS_PARSE_RATE() {
        @Override
        public String prettyName() {
            return "Parse/sec";
        }

        @Override
        public String prettyValue(double value) {
            return DataSize.shortString(Math.round(value));
        }

        @Override
        public double maximum(Collection<ShardSummary> shards, Collection<NodeSummary> nodes) {
            return evenlyAcrossCluster(this, shards, nodes);
        }
    },

    ;

    private static double evenlyAcrossCluster(
        Resource resource,
        Collection<ShardSummary> shards,
        Collection<NodeSummary> nodes)
    {
        long activeNodes = nodes.stream().map(NodeSummary::isActive).count();
        if (activeNodes == 0) {
            return 0;
        }

        double use = 0;
        for (var shard : shards) {
            use += shard.getResources().get(resource);
        }
        return Math.ceil(use / activeNodes);
    }
}
