package ru.yandex.solomon.balancer;

import java.util.HashMap;
import java.util.Map;

import ru.yandex.solomon.balancer.remote.RemoteNode;
import ru.yandex.solomon.balancer.remote.RemoteNodeState;
import ru.yandex.solomon.staffOnly.manager.special.DurationMillis;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * @author Vladimir Gordiychuk
 */
public class NodeSummary implements AutoCloseable {
    private final RemoteNode remote;
    private final Map<String, ShardSummary> shards = new HashMap<>();

    private NodeStatus status;
    private boolean active = true;
    private boolean freeze = false;
    private long expiredAt;

    @DurationMillis
    private long uptimeMillis;
    private long memoryBytes;
    private long utimeMillis;
    private final Resources current = new Resources();

    public NodeSummary(RemoteNode remote) {
        this.status = NodeStatus.UNKNOWN;
        this.remote = remote;
    }

    public RemoteNode getRemote() {
        return remote;
    }

    public void updateNodeState(RemoteNodeState state) {
        this.uptimeMillis = state.uptimeMillis;
        this.utimeMillis = state.utimeMillis;
        this.memoryBytes = state.memoryBytes;
        this.status = NodeStatus.CONNECTED;
    }

    public void setExpiredAt(long expiredAt) {
        this.expiredAt = expiredAt;
    }

    public void updateResources(Resources delta) {
        current.combine(delta);
    }

    public void markExpired() {
        status = NodeStatus.EXPIRED;
        utimeMillis = 0;
        memoryBytes = 0;
        uptimeMillis = 0;
    }

    public String getAddress() {
        return remote.getAddress();
    }

    public Map<String, ShardSummary> getShards() {
        return shards;
    }

    public boolean isEmpty() {
        return shards.isEmpty();
    }

    public boolean isActive() {
        return active;
    }

    public void setActive(boolean flag) {
        this.active = flag;
    }

    public boolean isFreeze() {
        return freeze;
    }

    public void setFreeze(boolean flag) {
        this.freeze = flag;
    }

    public long getUtimeMillis() {
        return utimeMillis;
    }

    public long getUptimeMillis() {
        return uptimeMillis;
    }

    public long getMemoryBytes() {
        return memoryBytes;
    }

    public long getExpiredAt() {
        return expiredAt;
    }

    public Resources getResources() {
        return current;
    }

    public Resource getDominantResource(Resources maximum) {
        return Resources.dominantResource(current, maximum);
    }

    public double getUsage(Resources maximum) {
        return Resources.max(Resources.normalize(current, maximum));
    }

    public NodeStatus getStatus() {
        return status;
    }

    public double getUsageWith(ShardSummary shard, Resources maximum) {
        final Resources next;
        if (shard.getNode() == this) {
            next = current;
        } else {
            next = new Resources(current)
                    .combine(shard.getResources());
        }

        return Resources.max(Resources.normalize(next, maximum));
    }

    public double getUsageWithout(ShardSummary shard, Resources maximum) {
        final Resources next;
        if (shard.getNode() != this) {
            next = current;
        } else {
            next = new Resources(current);
            next.minus(shard.getResources());
        }

        return Resources.max(Resources.normalize(next, maximum));
    }

    public void assign(ShardSummary shard, AssignmentSeqNo assignmentSeqNo) {
        if (shard.getNode() == this) {
            return;
        }

        shard.assign(this, assignmentSeqNo);
        current.plus(shard.getResources());
        shards.put(shard.getShardId(), shard);
    }

    public void successAssign(ShardSummary shard, long assignedAt) {
        if (shard.getNode() == this) {
            shard.setAssignedAt(assignedAt);
        }
    }

    public void failedAssign(ShardSummary shard, boolean globalFreeze) {
        if (shard.getNode() != this || globalFreeze || freeze) {
            return;
        }
        unassign(shard);
    }

    public void unassign(ShardSummary shard) {
        checkArgument(shards.containsKey(shard.getShardId()));
        current.minus(shard.getResources());
        shards.remove(shard.getShardId());
        if (shard.getNode() == this) {
            shard.unassign();
        }
    }

    public void unassignAll() {
        for (ShardSummary shard : shards.values()) {
            checkArgument(shard.getNode() == this);
            shard.unassign();
        }
        current.clear();
        shards.clear();
    }

    public double getFailCommandPercent() {
        return remote.getFailCommandPercent();
    }

    @Override
    public void close() {
        unassignAll();
        remote.close();
    }

    @Override
    public String toString() {
        return remote.getAddress();
    }
}
