package ru.yandex.solomon.balancer;

import java.util.Set;

import it.unimi.dsi.fastutil.objects.Object2DoubleOpenHashMap;

/**
 * @author Vladimir Gordiychuk
 */
public class Resources {
    public static final Resources EMPTY = new Resources();
    private final Object2DoubleOpenHashMap<Resource> map;

    public Resources() {
        this.map = new Object2DoubleOpenHashMap<>();
    }

    public Resources(Resources copy) {
        this.map = new Object2DoubleOpenHashMap<>(copy.map);
    }

    public void set(Resource resource, double value) {
        map.put(resource, value);
    }

    public void set(Resources resources) {
        map.putAll(resources.map);
    }

    public double get(Resource resource) {
        return map.getDouble(resource);
    }

    public void plus(Resources res) {
        var it = res.map.object2DoubleEntrySet().fastIterator();
        while (it.hasNext()) {
            var entry = it.next();
            map.addTo(entry.getKey(), entry.getDoubleValue());
        }
    }

    public void minus(Resources res) {
        var it = res.map.object2DoubleEntrySet().fastIterator();
        while (it.hasNext()) {
            var entry = it.next();
            map.addTo(entry.getKey(), -entry.getDoubleValue());
        }
    }

    public Resources combine(Resources res) {
        plus(res);
        return this;
    }

    public void clear() {
        map.clear();
    }

    public Set<Resource> keys() {
        return map.keySet();
    }

    public static Resources normalize(Resources res, Resources max) {
        Resources result = new Resources();
        for (var key : max.map.keySet()) {
            double maxForKey = max.get(key);
            if (maxForKey == 0.0) {
                result.set(key, 0);
            } else {
                result.set(key, res.get(key) / maxForKey);
            }
        }

        return result;
    }

    public static Resource dominantResource(Resources res, Resources max) {
        Resources normalized = Resources.normalize(res, max);
        Resource maxResource = CommonResource.SHARDS_COUNT;
        double maxValue = normalized.get(maxResource);
        var it = normalized.map.object2DoubleEntrySet().fastIterator();
        while (it.hasNext()) {
            var entry = it.next();
            if (Double.compare(maxValue, entry.getDoubleValue()) < 0) {
                maxResource = entry.getKey();
                maxValue = entry.getDoubleValue();
            }
        }
        return maxResource;
    }

    public static double max(Resources res) {
        double max = Double.MIN_VALUE;
        var it = res.map.values().iterator();
        while (it.hasNext()) {
            max = Double.max(it.nextDouble(), max);
        }
        return max;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        var that = (Resources) o;

        return map.equals(that.map);
    }

    @Override
    public int hashCode() {
        return map.hashCode();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Resources{");
        var it = map.object2DoubleEntrySet().fastIterator();
        while (it.hasNext()) {
            var entry = it.next();
            sb.append(entry.getKey().name()).append("=").append(entry.getDoubleValue()).append(", ");
        }
        if (sb.length() > 2) {
            sb.setLength(sb.length() - 2);
        }
        sb.append("}");
        return sb.toString();
    }
}
