package ru.yandex.solomon.balancer;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;

import ru.yandex.misc.lang.DefaultToString;
import ru.yandex.solomon.balancer.remote.RemoteShardState;
import ru.yandex.solomon.balancer.snapshot.SnapshotShard;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * @author Vladimir Gordiychuk
 */
@NotThreadSafe
public class ShardSummary extends DefaultToString {
    private static final long SHARD_INIT_TIME = TimeUnit.MINUTES.toMillis(10L);
    private final String shardId;
    private final Resources current;

    @Nullable
    private NodeSummary node;
    @Nullable
    private AssignmentSeqNo assignmentSeqNo;
    private ShardStatus status;
    private long uptimeMillis;
    private long assignedAt;
    private CompletableFuture<String> successAssignFuture;

    public ShardSummary(String shardId) {
        this.shardId = shardId;
        this.status = ShardStatus.UNKNOWN;
        this.current = new Resources();
        this.current.set(CommonResource.SHARDS_COUNT, 1);
        this.successAssignFuture = new CompletableFuture<>();
    }

    public String getShardId() {
        return shardId;
    }

    public ShardStatus getStatus() {
        return status;
    }

    public long getUptimeMillis() {
        if (assignedAt == 0) {
            return 0;
        }
        return uptimeMillis;
    }

    public long getAssignedAt() {
        return assignedAt;
    }

    public Resources getResources() {
        return current;
    }

    @Nullable
    public NodeSummary getNode() {
        return node;
    }

    public CompletableFuture<String> waitAssignment() {
        return successAssignFuture;
    }

    public void cancelAssignmentWaits(Throwable e) {
        successAssignFuture.completeExceptionally(e);
    }

    public void unassign() {
        this.node = null;
        if (this.successAssignFuture.isDone()) {
            this.successAssignFuture = new CompletableFuture<>();
        }
        this.assignmentSeqNo = null;
        this.status = ShardStatus.UNKNOWN;
        this.uptimeMillis = 0;
        this.assignedAt = 0;
    }

    public void assign(NodeSummary node, AssignmentSeqNo assignmentSeqNo) {
        this.node = node;
        this.status = ShardStatus.INIT;
        this.uptimeMillis = 0;
        this.assignmentSeqNo = assignmentSeqNo;
    }

    public void setAssignedAt(long assignedAt) {
        this.assignedAt = assignedAt;
        if (node != null) {
            successAssignFuture.complete(node.getAddress());
        }
    }

    public double getUsage(Resources maximum) {
        return Resources.max(Resources.normalize(current, maximum));
    }

    public AssignmentSeqNo getAssignmentSeqNo() {
        return assignmentSeqNo;
    }

    @SuppressWarnings("Duplicates")
    public void restoreResources(SnapshotShard shard) {
        checkArgument(Objects.equals(shardId, shard.shardId));
        current.set(shard.resources);
    }

    @SuppressWarnings("Duplicates")
    public Resources updateResources(RemoteShardState state, List<Resource> resources) {
        checkArgument(shardId.equals(state.shardId));
        status = state.status;
        uptimeMillis = state.uptimeMillis;
        if (!successAssignFuture.isDone()) {
            successAssignFuture.complete(Objects.requireNonNull(node).getAddress());
        }

        Resources prev = new Resources(current);
        current.set(state.resources);
        if (uptimeMillis < SHARD_INIT_TIME) {
            for (var resource : resources) {
                current.set(resource, Math.max(current.get(resource), prev.get(resource)));
            }
        }

        for (var resource : resources) {
            double delta = current.get(resource) - prev.get(resource);
            prev.set(resource, delta);
        }
        return prev;
    }
}
