package ru.yandex.solomon.balancer.dao;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.CompletableFuture;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import com.google.protobuf.ByteString;
import com.google.protobuf.Message;
import com.google.protobuf.UnsafeByteOperations;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.balancer.BalancerOptions;
import ru.yandex.solomon.balancer.BalancerProto;
import ru.yandex.solomon.balancer.protobuf.TBalancerAssignmentState;
import ru.yandex.solomon.balancer.protobuf.TBalancerOptions;
import ru.yandex.solomon.balancer.snapshot.SnapshotAssignments;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbBalancerDaoV2 implements BalancerDao {
    private static final Logger logger = LoggerFactory.getLogger(YdbBalancerDaoV2.class);

    private static final String OPTIONS_KEY = "TBalancerOptions";
    private static final String ASSIGNMENTS_KEY = "TBalancerAssignmentState";
    private final YdbBlobDao blobDao;

    public YdbBalancerDaoV2(String root, TableClient tableClient, SchemeClient schemeClient) {
        blobDao = new YdbBlobDao(root, "Balancer", tableClient, schemeClient);
    }

    @Override
    public CompletableFuture<Void> createSchema() {
        return blobDao.createSchema();
    }

    @Override
    public CompletableFuture<BalancerOptions> getOptions() {
        return blobDao.get(OPTIONS_KEY)
                .thenApply(bytes -> {
                    if (bytes.isEmpty()) {
                        logger.info("absent persistence options for balancer, use defaults");
                        return BalancerOptions.newBuilder().build();
                    }

                    TBalancerOptions proto = unzip(bytes.get(), TBalancerOptions::parseFrom);
                    return BalancerProto.fromProto(proto);
                });
    }

    @Override
    public CompletableFuture<Void> saveOptions(BalancerOptions opts) {
        var proto = BalancerProto.toProto(opts);
        return blobDao.put(OPTIONS_KEY, zip(proto));
    }

    @Override
    public CompletableFuture<SnapshotAssignments> getAssignments() {
        return blobDao.get(ASSIGNMENTS_KEY)
                .thenApply(bytes -> {
                    if (bytes.isEmpty()) {
                        logger.info("absent persistence state for balancer");
                        return SnapshotAssignments.EMPTY;
                    }

                    var proto = unzip(bytes.get(), TBalancerAssignmentState::parseFrom);
                    return BalancerProto.fromProto(proto);
                });
    }

    @Override
    public CompletableFuture<Void> saveAssignments(SnapshotAssignments assignments) {
        var proto = BalancerProto.toProto(assignments);
        return blobDao.put(ASSIGNMENTS_KEY, zip(proto));
    }

    private <T extends Message> ByteString zip(T message) {
        try (var out = new ByteArrayOutputStream();
             var zout = new GZIPOutputStream(out))
        {
            zout.write(message.toByteArray());
            zout.close();
            return UnsafeByteOperations.unsafeWrap(out.toByteArray());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private <T> T unzip(ByteString data, ParseFunction<T> fn) {
        try (var in = data.newInput();
             var zin = new GZIPInputStream(in);
        ) {
            return fn.apply(zin);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private interface ParseFunction<T> {
        T apply(InputStream stream) throws java.io.IOException;
    }

}
