package ru.yandex.solomon.balancer.dao;

import java.time.Duration;
import java.time.Instant;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;

import com.google.common.util.concurrent.MoreExecutors;
import com.google.protobuf.ByteString;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.transaction.TxControl;
import com.yandex.ydb.table.values.TupleValue;
import io.grpc.Status.Code;
import org.apache.commons.lang3.mutable.MutableInt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.balancer.dao.YdbBlobTable.ChunkPk;
import ru.yandex.solomon.balancer.dao.YdbBlobTable.ChunkReader;
import ru.yandex.solomon.balancer.dao.YdbBlobTable.ChunkRecord;
import ru.yandex.solomon.balancer.dao.YdbBlobTable.IndexColumnReader;
import ru.yandex.solomon.balancer.dao.YdbBlobTable.IndexRecord;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.protobuf.ByteStrings;

import static com.yandex.ydb.table.values.PrimitiveValue.string;
import static com.yandex.ydb.table.values.PrimitiveValue.timestamp;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbBlobDao {
    private static final Logger logger = LoggerFactory.getLogger(YdbBlobDao.class);
    private static final int CHUNK_SIZE = 4 << 20; // 4 MiB

    private final SessionRetryContext retryCtx;
    private final SchemeClient schemeClient;
    private final YdbBlobQuery query;

    public YdbBlobDao(String root, String prefix, TableClient tableClient, SchemeClient schemeClient) {
        this.schemeClient = schemeClient;
        this.query = new YdbBlobQuery(root, prefix);
        this.retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .sessionSupplyTimeout(Duration.ofSeconds(30))
                .build();
    }

    public CompletableFuture<Void> createSchema() {
        return schemeClient.makeDirectories(query.root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(unused -> createTable(query.tableChunk, YdbBlobTable::createChunkTable))
                .thenCompose(unused -> createTable(query.tableIndex, YdbBlobTable::createIndexTable));
    }

    private CompletableFuture<Void> createTable(String tablePath, BiFunction<String, Session, CompletableFuture<Status>> fn) {
        return schemeClient.describePath(tablePath)
                .thenCompose(exist -> {
                    if (exist.isSuccess()) {
                        return completedFuture(com.yandex.ydb.core.Status.SUCCESS);
                    }

                    return retryCtx.supplyStatus(session -> fn.apply(tablePath, session));
                })
                .thenAccept(status -> status.expect("cannot create create table " + tablePath));
    }

    public CompletableFuture<Void> put(String key, ByteString value) {
        ByteString[] chunks = ByteStrings.split(value, CHUNK_SIZE);

        var index = new IndexRecord(key, UUID.randomUUID().toString(), chunks.length, value.size(), Instant.now());
        return putChunks(index.chunkId(), chunks, index.createdAt())
                .thenCompose(ignore -> putIndex(index))
                .thenCompose(this::deleteChunks);
    }

    public CompletableFuture<Optional<ByteString>> get(String key) {
        var retryConfig = RetryConfig.DEFAULT
                .withNumRetries(3)
                .withExceptionFilter(throwable -> io.grpc.Status.fromThrowable(throwable).getCode() == Code.DATA_LOSS);

        return RetryCompletableFuture.runWithRetries(() -> {
            return getIndex(key)
                    .thenCompose(index -> {
                        if (index == null) {
                            return completedFuture(Optional.empty());
                        }

                        return getValue(index);
                    });
        }, retryConfig);
    }

    public CompletableFuture<Void> remove(String key) {
        var params = Params.of("$key", utf8(key), "$now", timestamp(Instant.now()));
        return execute(query.deleteIndex, params)
                .thenApply(result -> {
                    var rs = result.expect("Unable delete " + key).getResultSet(0);
                    if (rs.next()) {
                        var reader = new IndexColumnReader(rs);
                        return reader.read(rs);
                    }

                    return null;
                })
                .thenCompose(this::deleteChunks);
    }

    private boolean isChunksConsistence(IndexRecord index, List<ChunkRecord> chunks) {
        for (var idx = 0; idx < chunks.size(); idx++) {
            if (chunks.get(idx).index() != idx) {
                return false;
            }
        }

        return chunks.size() == index.chunkCount();
    }

    private CompletableFuture<Optional<ByteString>> getValue(IndexRecord index) {
        if (index.chunkCount() == 0) {
            return completedFuture(Optional.of(ByteString.EMPTY));
        }

        return getChunksAsReadTable(index)
                .thenApply(chunks -> {
                    if (!isChunksConsistence(index, chunks)) {
                        throw io.grpc.Status.DATA_LOSS
                                .withDescription("Chunks not consistent for " + index)
                                .asRuntimeException();
                    }

                    return chunks.stream().map(ChunkRecord::value).reduce(ByteString::concat);
                });
    }

    private CompletableFuture<List<ChunkRecord>> getChunksAsReadTable(IndexRecord index) {
        var fromKey = TupleValue.of(
                utf8(index.chunkId()).makeOptional(),
                uint32(0).makeOptional());

        var toKey = TupleValue.of(
                utf8(index.chunkId()).makeOptional(),
                uint32(index.chunkCount()).makeOptional());

        var reader = new ChunkReader();
        var read = retryCtx.supplyStatus(session -> {
            var settings = ReadTableSettings.newBuilder()
                    .orderedRead(true)
                    .timeout(1, TimeUnit.MINUTES)
                    .toKeyExclusive(toKey);

            var lastKey = reader.lastKey();
            if (lastKey != null) {
                settings.fromKeyExclusive(lastKey);
            } else {
                settings.fromKeyInclusive(fromKey);
            }

            return session.readTable(query.tableChunk, settings.build(), reader);
        });

        return read.thenAccept(status -> status.expect("can't read " + query.tableChunk))
                .thenApply(unused -> {
                    reader.chunks.sort(Comparator.comparingInt(ChunkRecord::index));
                    return reader.chunks;
                });
    }

    private CompletableFuture<IndexRecord> getIndex(String key) {
        var params = Params.of("$key", utf8(key));
        return execute(query.selectIndex, params)
                .thenApply(result -> {
                    var rs = result.expect("can not read index").getResultSet(0);
                    if (!rs.next()) {
                        return null;
                    }

                    var reader = new IndexColumnReader(rs);
                    return reader.read(rs);
                });
    }

    private CompletableFuture<Void> deleteChunks(IndexRecord record) {
        if (record == null || record.chunkCount() == 0) {
            return completedFuture(null);
        }

        MutableInt idx = new MutableInt(0);
        AsyncActorBody body = () -> {
            if (idx.intValue() == record.chunkCount()) {
                return completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var primaryKey = new ChunkPk(record.chunkId(), idx.getAndIncrement());
            return deleteChunk(primaryKey);
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, MoreExecutors.directExecutor(), 2);
        return runner.start();
    }

    private CompletableFuture<Void> deleteChunk(ChunkPk primaryKey) {
        var params = Params.of(
                "$id", utf8(primaryKey.id()),
                "$index", uint32(primaryKey.index()));
        return execute(query.deleteChunk, params)
                .thenAccept(result -> result.expect("Unable delete chunk " + primaryKey))
                .exceptionally(e -> {
                    logger.warn("Unable delete chunk {}", primaryKey, e);
                    return null;
                });
    }

    private CompletableFuture<IndexRecord> putIndex(IndexRecord record) {
        var params = Params.create()
                .put("$key", utf8(record.key()))
                .put("$chunk_id", utf8(record.chunkId()))
                .put("$chunk_count", uint32(record.chunkCount()))
                .put("$chunk_bytes", uint32(record.chunkBytes()))
                .put("$now", timestamp(record.createdAt()));

        return execute(query.insertIndex, params)
                .thenApply(result -> {
                    var rs = result.expect("unable insert index").getResultSet(0);
                    if (rs.next()) {
                        var reader = new IndexColumnReader(rs);
                        return reader.read(rs);
                    }

                    return null;
                });
    }

    private CompletableFuture<Void> putChunks(String id, ByteString[] chunks, Instant now) {
        var root = new CompletableFuture<Void>();
        var future = root;
        for (int index = 0; index < chunks.length; index++) {
            var record = new ChunkRecord(id, index, chunks.length, now, now, chunks[index]);
            future = future.thenCompose(ignore -> putChunk(record));
        }
        root.complete(null);
        return future;
    }

    private CompletableFuture<Void> putChunk(ChunkRecord record) {
        var params = Params.create()
                .put("$id", utf8(record.id()))
                .put("$index", uint32(record.index()))
                .put("$count", uint32(record.count()))
                .put("$now", timestamp(record.createAt()))
                .put("$value", string(record.value()));

        return execute(query.insertChunk, params)
                .thenAccept(result -> {
                    result.expect("Unable insert chunk");
                });
    }

    private CompletableFuture<Result<DataQueryResult>> execute(String query, Params params) {
        try {
            return retryCtx.supplyResult(s -> {
                var settings = new ExecuteDataQuerySettings().keepInQueryCache();
                var tx = TxControl.serializableRw();
                return s.executeDataQuery(query, tx, params, settings);
            });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }
}
