package ru.yandex.solomon.balancer.dao;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.annotation.Nullable;

import com.google.protobuf.ByteString;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.AutoPartitioningPolicy;
import com.yandex.ydb.table.settings.CreateTableSettings;
import com.yandex.ydb.table.settings.PartitioningPolicy;
import com.yandex.ydb.table.settings.TtlSettings;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.TupleValue;

import static ru.yandex.solomon.ydb.YdbResultSets.bytes;
import static ru.yandex.solomon.ydb.YdbResultSets.timestamp;
import static ru.yandex.solomon.ydb.YdbResultSets.uint32;
import static ru.yandex.solomon.ydb.YdbResultSets.utf8;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbBlobTable {

    static CompletableFuture<Status> createIndexTable(String tablePath, Session session) {
        var table = TableDescription.newBuilder()
                .addNullableColumn("key", PrimitiveType.utf8())
                .addNullableColumn("chunk_id", PrimitiveType.utf8())
                .addNullableColumn("chunk_count", PrimitiveType.uint32())
                .addNullableColumn("chunk_bytes", PrimitiveType.uint32())
                .addNullableColumn("created_at", PrimitiveType.timestamp())
                .setPrimaryKeys("key")
                .build();

        var settings = new CreateTableSettings();
        settings.setPartitioningPolicy(new PartitioningPolicy().setAutoPartitioning(AutoPartitioningPolicy.AUTO_SPLIT_MERGE));
        settings.setTimeout(10, TimeUnit.SECONDS);

        return session.createTable(tablePath, table, settings);
    }

    static CompletableFuture<Status> createChunkTable(String tablePath, Session session) {
        var table = TableDescription.newBuilder()
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("index", PrimitiveType.uint32())
                .addNullableColumn("count", PrimitiveType.uint32())
                .addNullableColumn("created_at", PrimitiveType.timestamp())
                .addNullableColumn("deleted_at", PrimitiveType.timestamp())
                .addNullableColumn("value", PrimitiveType.string())
                .setPrimaryKeys("id", "index")
                .build();

        var settings = new CreateTableSettings();
        settings.setTtlSettings(new TtlSettings("deleted_at", (int) TimeUnit.MINUTES.toSeconds(30)));
        settings.setPartitioningPolicy(new PartitioningPolicy().setAutoPartitioning(AutoPartitioningPolicy.AUTO_SPLIT_MERGE));
        settings.setTimeout(10, TimeUnit.SECONDS);

        return session.createTable(tablePath, table, settings);
    }

    record ChunkRecord(String id, int index, int count, Instant createAt, Instant deletedAt, ByteString value) {
    }

    record ChunkPk(String id, int index) {
    }

    static class ChunkColumnReader {
        private final int idIdx;
        private final int indexIdx;
        private final int countIdx;
        private final int createAtIdx;
        private final int deletedAtIdx;
        private final int valueIdx;

        public ChunkColumnReader(ResultSetReader resultSet) {
            this.idIdx = resultSet.getColumnIndex("id");
            this.indexIdx = resultSet.getColumnIndex("index");
            this.countIdx = resultSet.getColumnIndex("count");
            this.createAtIdx = resultSet.getColumnIndex("created_at");
            this.deletedAtIdx = resultSet.getColumnIndex("deleted_at");
            this.valueIdx = resultSet.getColumnIndex("value");
        }

        public ChunkRecord read(ResultSetReader rs) {
            return new ChunkRecord(
                    utf8(rs, idIdx),
                    uint32(rs, indexIdx),
                    uint32(rs, countIdx),
                    timestamp(rs, createAtIdx),
                    timestamp(rs, deletedAtIdx),
                    bytes(rs, valueIdx));
        }
    }

    static class ChunkReader implements Consumer<ResultSetReader> {
        final List<ChunkRecord> chunks = new ArrayList<>();

        @Override
        public void accept(ResultSetReader rs) {
            int rowCount = rs.getRowCount();
            if (rowCount == 0) {
                return;
            }

            var columns = new ChunkColumnReader(rs);
            while (rs.next()) {
                chunks.add(columns.read(rs));
            }
        }

        @Nullable
        public TupleValue lastKey() {
            if (chunks.isEmpty()) {
                return null;
            }

            var last = chunks.get(chunks.size() - 1);
            return TupleValue.of(
                    PrimitiveValue.utf8(last.id).makeOptional(),
                    PrimitiveValue.uint32(last.index).makeOptional());
        }
    }

    record IndexRecord(String key, String chunkId, int chunkCount, int chunkBytes, Instant createdAt){
    }

    static class IndexColumnReader {
        private final int keyIdx;
        private final int chunkIdIdx;
        private final int chunkCountIdx;
        private final int chunkBytesIdx;
        private final int createdAtIdx;

        public IndexColumnReader(ResultSetReader resultSet) {
            this.keyIdx = resultSet.getColumnIndex("key");
            this.chunkIdIdx = resultSet.getColumnIndex("chunk_id");
            this.chunkCountIdx = resultSet.getColumnIndex("chunk_count");
            this.chunkBytesIdx = resultSet.getColumnIndex("chunk_bytes");
            this.createdAtIdx = resultSet.getColumnIndex("created_at");
        }

        public IndexRecord read(ResultSetReader rs) {
            return new IndexRecord(
                    utf8(rs, keyIdx),
                    utf8(rs, chunkIdIdx),
                    uint32(rs, chunkCountIdx),
                    uint32(rs, chunkBytesIdx),
                    timestamp(rs, createdAtIdx));
        }
    }
}
