package ru.yandex.solomon.balancer.remote;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.Nullable;

import com.google.common.util.concurrent.MoreExecutors;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.monlib.metrics.meter.ExpMovingAverage;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.solomon.util.actors.PingActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class AbstractRemoteNode implements RemoteNode {
    private static final Logger logger = LoggerFactory.getLogger(AbstractRemoteNode.class);

    protected final String address;
    protected final long leaderSeqNo;

    protected final ScheduledExecutorService timer;
    protected final Clock clock;

    private final Meter successMessage = Meter.of(ExpMovingAverage.oneMinute());
    private final Meter failMessage = Meter.of(ExpMovingAverage.oneMinute());
    private final AtomicReference<RemoteNodeState> state = new AtomicReference<>();
    private final PingActorRunner actor;

    // for manager ui
    private Throwable lastError;
    private Instant lastErrorTime;

    public AbstractRemoteNode(String address, long leaderSeqNo, ScheduledExecutorService timer, Clock clock) {
        this.address = address;
        this.leaderSeqNo = leaderSeqNo;
        this.timer = timer;
        this.clock = clock;
        this.actor = PingActorRunner.newBuilder()
                .operation("ping_node_" + address)
                .pingInterval(Duration.ofSeconds(5))
                .backoffDelay(Duration.ofSeconds(1))
                .backoffMaxDelay(Duration.ofSeconds(15))
                .executor(MoreExecutors.directExecutor())
                .timer(timer)
                .onPing(this::act)
                .build();
        actor.schedule();
    }

    // TODO: remote after migrate all to ping
    public void receiveHeartbeat(RemoteNodeState remoteState) {
        state.set(remoteState);
        successMessage.mark();
    }

    @Override
    public String getAddress() {
        return address;
    }

    @Nullable
    @Override
    public RemoteNodeState takeState() {
        return state.getAndSet(null);
    }

    @Override
    public double getFailCommandPercent() {
        double failRate = failMessage.getRate(TimeUnit.SECONDS);
        if (Double.compare(failRate, 0d) == 0) {
            return 0d;
        }

        double successRate = successMessage.getRate(TimeUnit.SECONDS);
        return failRate / (successRate + failRate);
    }

    private CompletableFuture<?> act(int attempt) {
        return ping()
                .thenApply(remoteState -> {
                    var remoteAddress = Optional.ofNullable(remoteState).map(r -> r.address).orElse(address);
                    if (!address.equals(remoteAddress)) {
                        var status = Status.FAILED_PRECONDITION.withDescription("Mismatch remote address " + remoteAddress + " != node address " + address);
                        throw new StatusRuntimeExceptionNoStackTrace(status);
                    }
                    return remoteState;
                })
                .whenComplete((remoteState, error) -> {
                    if (error != null) {
                        saveLastError(error);
                        var status = Status.fromThrowable(error);
                        if (status.getCode() != Status.Code.UNIMPLEMENTED) {
                            failMessage.mark();
                        }
                    } else {
                        successMessage.mark();
                        if (remoteState != null) {
                            state.set(remoteState);
                        }
                    }
                });
    }

    protected abstract CompletableFuture<RemoteNodeState> ping();

    protected void trackCall(CompletableFuture<?> future) {
        future.whenComplete((ignore, e) -> {
            if (e != null) {
                failMessage.mark();
                saveLastError(e);
            } else {
                successMessage.mark();
            }
        });
    }

    private void saveLastError(Throwable t) {
        lastError = t;
        lastErrorTime = clock.instant();
    }

    @Override
    public void close() {
        actor.close();
    }
}
