package ru.yandex.solomon.balancer.remote;

import java.time.Clock;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;

import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Vladimir Gordiychuk
 */
public class RemoteClusterImpl implements RemoteCluster {
    private static final Logger logger = LoggerFactory.getLogger(RemoteClusterImpl.class);

    private final RemoteNodeClient client;
    private final ScheduledExecutorService timer;
    private final Clock clock;
    private final ConcurrentHashMap<String, RemoteNodeImpl> nodeByAddress = new ConcurrentHashMap<>();

    public RemoteClusterImpl(RemoteNodeClient client, ScheduledExecutorService timer, Clock clock) {
        this.client = client;
        this.timer = timer;
        this.clock = clock;
    }

    @Override
    public boolean hasNode(String node) {
        return client.getNodes().contains(node);
    }

    @Override
    public Set<String> getNodes() {
        return client.getNodes();
    }

    @Override
    public void receiveHeartbeat(RemoteNodeState state) {
        var node = nodeByAddress.get(state.address);
        logger.debug("Client has node {}: {}", state.address, client.hasNode(state.address));
        logger.debug("nodeByAddress {} present: {}", state.address, node != null);
        if (!client.hasNode(state.address) || node == null) {
            throw Status.FAILED_PRECONDITION
                    .withDescription("Unknown node: " + state.address)
                    .asRuntimeException();
        }

        node.receiveHeartbeat(state);
    }

    @Override
    public RemoteNode create(String address, long leaderSeqNo) {
        var node = new RemoteNodeImpl(address, leaderSeqNo, timer, clock, client);
        nodeByAddress.compute(address, (s, prev) -> {
            logger.debug("Register remote node {}: prev.leaderSeqNo = {}, newNode.leaderSeqNo = {}",
                    address,
                    Optional.ofNullable(prev).map(remoteNode -> remoteNode.leaderSeqNo).orElse(-1L),
                    node.leaderSeqNo);
            if (prev == null || prev.leaderSeqNo <= node.leaderSeqNo) {
                return node;
            }
            return prev;
        });
        return node;
    }
}
