package ru.yandex.solomon.balancer.www;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.balancer.Balancer;
import ru.yandex.solomon.balancer.BalancerHolder;
import ru.yandex.solomon.balancer.NodeStatus;
import ru.yandex.solomon.balancer.NodeSummary;
import ru.yandex.solomon.balancer.ShardStatus;
import ru.yandex.solomon.balancer.ShardSummary;
import ru.yandex.solomon.staffOnly.RedirectException;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.util.collection.Nullables;

import static ru.yandex.solomon.staffOnly.manager.ManagerController.namedObjectLink;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
public class BalancerController {
    private final HttpAuthenticator authenticator;
    private final InternalAuthorizer authorizer;
    private final BalancerHolder balancerHolder;
    private final Class<?> shardClass;
    private final int port;

    public BalancerController(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            BalancerHolder balancerHolder,
            Class<?> shardClass,
            int port) {
        this.authenticator = authenticator;
        this.authorizer = authorizer;
        this.balancerHolder = balancerHolder;
        this.shardClass = shardClass;
        this.port = port;
    }

    @RequestMapping(value = "/balancer", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> balancer(ServerHttpRequest request) {
        return authorize(request)
            .thenApply(account -> balancerImpl(request));
    }

    private String balancerImpl(ServerHttpRequest request) {
        var leader = getBalancer(request);
        int sortBy = Integer.parseInt(Nullables.orDefault(request.getQueryParams().getFirst("sortBy"), "1"));
        return new BalancerPageWww(leader, sortBy, port).genString();
    }

    @RequestMapping(value = "/balancer/ok", produces = MediaType.TEXT_PLAIN_VALUE)
    public CompletableFuture<ResponseEntity<String>> balancerOk(ServerHttpRequest request) {
        return authorize(request)
            .thenApply(account -> balancerOkImpl(request));
    }

    private ResponseEntity<String> balancerOkImpl(ServerHttpRequest request) {
        var leader = getBalancer(request);

        var shards = leader.getShards();
        String[] notReadyShards = shards.values().stream()
            .filter(shardSummary -> shardSummary.getStatus() != ShardStatus.READY)
            .map(ShardSummary::getShardId)
            .toArray(String[]::new);

        if (notReadyShards.length > 0) {
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR)
                .body("Ready shards " + notReadyShards.length + "/" + shards.size() + " : " + Arrays.toString(notReadyShards));
        }

        var nodes = new ArrayList<>(leader.getNodes().values());
        var unknownNodes = nodes.stream()
            .filter(summary -> summary.getStatus() != NodeStatus.CONNECTED)
            .map(NodeSummary::getAddress)
            .collect(Collectors.toList());

        if (unknownNodes.size() > 0) {
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR)
                .body("Ready nodes " + unknownNodes.size() + "/" + nodes.size() + " : " + unknownNodes);
        }

        return ResponseEntity.ok().body("");
    }

    @RequestMapping(value = "/balancer/settings", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> setting(ServerHttpRequest request) {
        return authorize(request)
            .thenApply(account -> settingImpl(request));
    }

    private String settingImpl(ServerHttpRequest request) {
        var balancer = getBalancer(request);
        return new BalancerSettingsPageWww(balancer).genString();
    }

    @RequestMapping(value = "/balancer/who", produces = MediaType.TEXT_PLAIN_VALUE)
    public CompletableFuture<ResponseEntity<String>> whoIsLeader(ServerHttpRequest request) {
        return authorize(request)
                .thenApply(account -> {
                    var leader = balancerHolder.getLeaderNode();
                    if (Strings.isNullOrEmpty(leader)) {
                        return ResponseEntity.status(HttpStatus.NOT_FOUND)
                                .body("");
                    }

                    return ResponseEntity.ok(leader);
                });
    }

    @RequestMapping("/balancer/settings/save")
    public CompletableFuture<?> saveSetting(ServerHttpRequest request) {
        return authorize(request)
                .thenCompose(account -> {
                    var balancer = getBalancer(request);
                    var opts = BalancerSettingsPageWww.parseOptions(request, balancer.getResources());
                    return balancer.setOptions(opts);
                })
                .thenApply(ignore -> {
                    throw new RedirectException("/balancer");
                });
    }

    @RequestMapping("/balancer/rebalance")
    public CompletableFuture<String> rebalance(ServerHttpRequest request) {
        return authorize(request)
            .thenCompose(account -> rebalanceImpl(request));
    }

    private CompletableFuture<String> rebalanceImpl(ServerHttpRequest request) {
        var balancer = getBalancer(request);
        return balancer.rebalance().thenApply(ignore -> redirectBack(request));
    }

    @RequestMapping("/balancer/cancelRebalance")
    public CompletableFuture<String> cancelRebalance(ServerHttpRequest request) {
        return authorize(request)
            .thenCompose(account -> cancelRebalanceImpl(request));
    }

    private CompletableFuture<String> cancelRebalanceImpl(ServerHttpRequest request) {
        var balancer = getBalancer(request);
        return balancer.cancelRebalance().thenApply(ignore -> redirectBack(request));
    }

    @RequestMapping("/balancer/kickShard")
    public CompletableFuture<String> kickShard(
        @RequestParam(value = "shardId") String shardId,
        ServerHttpRequest request)
    {
        return authorize(request)
            .thenCompose(account -> kickShardImpl(shardId, request));
    }

    private CompletableFuture<String> kickShardImpl(String shardId, ServerHttpRequest request) {
        var balancer = getBalancer(request);
        return balancer.kickShard(shardId)
            .thenApply(ignore -> redirectBack(request));
    }

    @RequestMapping("/balancer/kickNode")
    public CompletableFuture<?> kickNode(
        @RequestParam(value = "node") String address,
        ServerHttpRequest request)
    {
        return authorize(request)
            .thenCompose(account -> kickNodeImpl(address, request));
    }

    private CompletableFuture<?> kickNodeImpl(String address, ServerHttpRequest request) {
        var balancer = getBalancer(request);
        return balancer.kickNode(address)
            .thenApply(ignore -> redirectBack(request));
    }

    @RequestMapping("/balancer/nodeActive")
    public CompletableFuture<?> setActiveFlag(
        @RequestParam(value = "node") String address,
        @RequestParam(value = "flag") boolean flag,
        ServerHttpRequest request)
    {
        return authorize(request)
            .thenCompose(account -> setActiveFlagImpl(address, flag, request));
    }

    private CompletableFuture<?> setActiveFlagImpl(String address, boolean flag, ServerHttpRequest request) {
        var balancer = getBalancer(request);
        return balancer.setActive(address, flag)
            .thenApply(ignore -> redirectBack(request));
    }

    @RequestMapping("/balancer/nodeFreeze")
    public CompletableFuture<?> setFreeze(
        @RequestParam(value = "node") String address,
        @RequestParam(value = "flag") boolean flag,
        ServerHttpRequest request)
    {
        return authorize(request)
            .thenCompose(account -> setFreezeImpl(address, flag, request));
    }

    private CompletableFuture<?> setFreezeImpl(String address, boolean flag, ServerHttpRequest request) {
        var balancer = getBalancer(request);
        return balancer.setFreeze(address, flag)
            .thenApply(ignore -> redirectBack(request));
    }

    @RequestMapping("/balancer/allNodeActive")
    public CompletableFuture<?> setActiveFlag(
        @RequestParam(value = "flag") boolean flag,
        ServerHttpRequest request)
    {
        return authorize(request)
                .thenCompose(account -> setActiveFlagImpl(flag, request));
        }

    private CompletableFuture<?> setActiveFlagImpl(boolean flag, ServerHttpRequest request) {
        var balancer = getBalancer(request);
        return balancer.setActive(flag)
            .thenApply(ignore -> redirectBack(request));
    }

    @RequestMapping("/balancer/allNodeFreeze")
    public CompletableFuture<?> setFreezeFlag(
        @RequestParam(value = "flag") boolean flag,
        ServerHttpRequest request)
    {
        return authorize(request)
            .thenCompose(account -> setFreezeFlagImpl(flag, request));
    }

    private CompletableFuture<?> setFreezeFlagImpl(boolean flag, ServerHttpRequest request) {
        var balancer = getBalancer(request);
        return balancer.setFreeze(flag)
            .thenApply(ignore -> redirectBack(request));
    }

    @RequestMapping(path = "/balancer/shards/{shardId}", method = RequestMethod.GET)
    public CompletableFuture<ResponseEntity<String>> redirectToShard(
        @PathVariable("shardId") String shardId,
        ServerHttpRequest request)
    {
        return authorize(request)
                .thenApply(ignore -> getBalancer(request))
                .thenCompose(leader -> leader.getAssignment(shardId))
                .thenApply(host -> {
                    if (Strings.isNullOrEmpty(host)) {
                        return ResponseEntity.status(HttpStatus.NOT_FOUND)
                                .body("Shard " + shardId + " not assigned on node yet, try again later");
                    }

                    String linkToShard = namedObjectLink(new NamedObjectId(shardClass, shardId));
                    String url = "http://" + host + ":" + port + "" + linkToShard;

                    return ResponseEntity.status(HttpStatus.FOUND)
                            .header("Location", url)
                            .body(null);
                });
    }

    private CompletableFuture<Account> authorize(ServerHttpRequest request) {
        return authenticator.authenticate(request)
                .thenCompose(authorizer::authorize);
    }

    private String redirectBack(ServerHttpRequest request) {
        String referer = request.getHeaders().getFirst("Referer");
        if (Strings.isNullOrEmpty(referer)) {
            throw new RedirectException("/balancer");
        }

        throw new RedirectException(referer);
    }

    private String targetUrl(ServerHttpRequest request) {
        var uri = request.getURI();
        return Strings.isNullOrEmpty(uri.getQuery())
                ? uri.getPath()
                : uri.getPath() + "?" + uri.getQuery();
    }

    private <T> T redirectToLeader(String path) {
        var leader = balancerHolder.getLeaderNode();
        if (Strings.isNullOrEmpty(leader)) {
            throw new IllegalStateException("leader unknown, try again later");
        }
        throw new RedirectException("http://" + leader + ":" + port + path);
    }

    public Balancer getBalancer(ServerHttpRequest req) {
        var balancer = balancerHolder.getBalancer();
        if (balancer == null) {
            return redirectToLeader(targetUrl(req));
        }
        return balancer;
    }
}
