package ru.yandex.solomon.balancer.www;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.IntConsumer;

import ru.yandex.solomon.balancer.Balancer;
import ru.yandex.solomon.balancer.CommonResource;
import ru.yandex.solomon.balancer.NodeStatus;
import ru.yandex.solomon.balancer.NodeSummary;
import ru.yandex.solomon.balancer.Resources;
import ru.yandex.solomon.balancer.ShardStatus;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class BalancerPageWww extends ManagerPageTemplate {
    private final Balancer balancer;
    private final int sortBy;
    private final int port;

    public BalancerPageWww(Balancer balancer, int sortBy, int port) {
        super("Shards balancer");
        this.balancer = balancer;
        this.sortBy = sortBy;
        this.port = port;
    }

    @Override
    public void commonCssJs() {
        super.commonCssJs();
        tag("style", () -> {
            writeRaw(".table-hover tbody tr:hover > td { background-color: #9dddf2 }");
        });
    }

    @Override
    protected void content() {
        List<NodeSummary> rows = new ArrayList<>(balancer.getNodes().values());
        long connectedCount = rows.stream()
            .filter(node -> node.getStatus() == NodeStatus.CONNECTED)
            .count();

        Resources totalRes = rows.stream()
            .map(NodeSummary::getResources)
            .reduce(new Resources(), Resources::combine);

        h2("Links");
        ul(() -> {
            li(() -> aHref("/balancer/settings", "Settings"));
        });

        h2("Summary");
        tag("table.simple-table1", () -> {
            tr(() -> {
                thText("Nodes: ");
                tdText(connectedCount + " / " + rows.size());
            });
            tr(() -> {
                thText("Health: ");
                td(() -> {
                    if (connectedCount == rows.size()) {
                        label("success", "100%");
                    } else {
                        label("warning", Math.round(100.0 * ((double) connectedCount / rows.size())) + "%");
                    }
                });
            });
            tr(() -> {
                thText("Dispersion: ");
                tdText(String.format("%.5f", balancer.getDispersion()));
            });
            tr(() -> {
                th(() -> {
                    aHref("/balancer/rebalance", "Rebalance: ");
                });
                double progress = balancer.getRebalanceProgress();
                if (progress < 1.0) {
                    td(() -> {
                        progress(progress * 100);
                    });
                    td(() -> {
                        aHref("/balancer/cancelRebalance", () -> {
                            tag("span", HtmlWriter.Attr.cssClass("glyphicon glyphicon-remove"));
                        });
                    });
                }
            });
            for (var resource : balancer.getResources()) {
                if (resource != CommonResource.SHARDS_COUNT) {
                    tr(() -> {
                        thText(resource.prettyName() + ": ");
                        tdText(resource.prettyValue(totalRes.get(resource)));
                    });
                } else {
                    thText(resource.prettyName() + ": ");
                    var shards = balancer.getShards();
                    long total = shards.size();
                    long ready = shards.values()
                            .stream()
                            .filter(shard -> {
                                if (shard.getNode() == null) {
                                    return false;
                                }
                                return shard.getStatus() == ShardStatus.READY;
                            }).count();
                    tdText(resource.prettyValue(ready) + " / " + resource.prettyValue(total));
                }
            }

            String autoFreezeReason = balancer.getAutoFreezeReason();
            if (autoFreezeReason != null) {
                tr(() -> {
                    thText("AutoFreeze: ");
                    tdText(autoFreezeReason);
                });
            }
        });

        h2("Nodes");
        writeNodeSummaryTable(balancer.getResourceMaximum(), rows);
    }

    private void writeNodeSummaryTable(Resources maximum, List<NodeSummary> rows) {
        List<Column> columns = makeNodeSummaryColumns(maximum, rows);
        rows.sort(columns.get(0).comparator);
        rows.sort(columns.get(Math.abs(sortBy) - 1).comparator);
        if (sortBy < 0) {
            Collections.reverse(rows);
        }

        tag("table", () -> {
            tr(() -> {
                for (int index = 0; index < columns.size(); index++) {
                    int i = index;
                    th(() -> {
                        columns.get(i).header.accept(i);
                    });
                }
            });

            for (NodeSummary node : rows) {
                if (node == null) {
                    continue;
                }
                tr(() -> {
                    for (int index = 0; index < columns.size(); index++) {
                        try {
                            var column = columns.get(index);
                            td(() -> {
                                column.content.accept(node);
                            });
                        } catch (Throwable e) {
                            throw new RuntimeException("Failed at column " + index + " on node " + node, e);
                        }
                    }
                });
            }
        }, Attr.cssClass("table simple-table2 table-hover table-condensed"));
    }

    public static String formatDuration(long millis) {
        return DurationUtils.formatDurationMillisTruncated(millis);
    }

    private List<Column> makeNodeSummaryColumns(Resources maximum, List<NodeSummary> rows) {
        boolean anyActive = rows.stream().anyMatch(NodeSummary::isActive);
        boolean anyFreeze = rows.stream().anyMatch(NodeSummary::isFreeze);
        List<Column> columns = new ArrayList<>();

        columns.add(column("Node", node -> {
            String address = node.getAddress();
            if (HostUtils.getFqdn().equals(address)) {
                tag("span", HtmlWriter.Attr.cssClass("glyphicon glyphicon-star"));
            } else {
                tag("span", HtmlWriter.Attr.cssClass("glyphicon"));
            }
            write(" ");
            aHref("http://" + address + ":" + port + "/local-shards", HostUtils.shortName(address));
        }, Comparator.comparing(NodeSummary::getAddress)));

        columns.add(column("Status", node -> {
            switch (node.getStatus()) {
                case CONNECTED:
                    label("success", "connected");
                    break;
                case EXPIRED:
                    label("info", "expired");
                    break;
                case UNKNOWN:
                    label("default", "unknown");
                    break;
            }
        }, Comparator.comparing(NodeSummary::getStatus)));

        columns.add(column("Uptime", node -> {
            write(formatDuration(node.getUptimeMillis()));
        }, Comparator.comparingLong(NodeSummary::getUptimeMillis)));

        columns.add(column("Expire", node -> {
            write(formatDuration(node.getExpiredAt() - System.currentTimeMillis()));
        }, Comparator.comparingLong(NodeSummary::getExpiredAt)));

        columns.add(column("Usage", node -> {
            write(String.format("%.5f", node.getUsage(maximum)));
        }, Comparator.comparingDouble(node -> node.getUsage(maximum))));

        for (var resource : balancer.getResources()) {
            if (resource != CommonResource.SHARDS_COUNT) {
                columns.add(column(resource.prettyName(), node -> {
                    var dominant = node.getDominantResource(maximum);
                    var value = resource.prettyValue(node.getResources().get(resource));
                    if (dominant == resource) {
                        b(value);
                    } else {
                        write(value);
                    }
                }, Comparator.comparingDouble(node -> node.getResources().get(resource))));
            } else {
                columns.add(column(resource.prettyName(), node -> {
                    var dominant = node.getDominantResource(maximum);
                    int count = (int) Math.round(node.getResources().get(resource));
                    int ready = (int) node.getShards()
                            .values()
                            .stream()
                            .filter(shard -> shard.getStatus() == ShardStatus.READY)
                            .count();

                    String value = ready != count
                            ? ready + " / " + count
                            : String.valueOf(count);

                    if (dominant == resource) {
                        b(value);
                    } else {
                        write(value);
                    }
                }, Comparator.comparingDouble(node -> node.getResources().get(resource))));
            }
        }

        columns.add(column("Fails", node -> {
            write(String.format("%.3f", node.getFailCommandPercent()));
        }, Comparator.comparingDouble(NodeSummary::getFailCommandPercent)));

        columns.add(column(ignore -> {
            aHref("/balancer/allNodeActive?flag=" + !anyActive, "Active");
        }, node -> {
            boolean active = node.isActive();
            aHref("/balancer/nodeActive?node=" + node.getAddress() + "&flag=" + !active, () -> {
                if (active) {
                    tag("span", HtmlWriter.Attr.cssClass("active-mark glyphicon glyphicon-ok"));
                } else {
                    tag("span", HtmlWriter.Attr.cssClass("active-mark glyphicon glyphicon-remove"));
                }
            });
        }));

        columns.add(column(ignore -> {
            aHref("/balancer/allNodeFreeze?flag=" + !anyFreeze, "Freeze");
        }, node -> {
            boolean freeze = node.isFreeze();
            aHref("/balancer/nodeFreeze?node=" + node.getAddress() + "&flag=" + !freeze, () -> {
                if (freeze) {
                    tag("span", HtmlWriter.Attr.cssClass("glyphicon glyphicon-pause"));
                } else {
                    tag("span", HtmlWriter.Attr.cssClass("glyphicon glyphicon-play"));
                }
            });
        }));

        columns.add(column(ignore -> {
            write("Kick");
        }, node -> {
            aHref("/balancer/kickNode?node=" + node.getAddress(), () -> {
                tag("span", HtmlWriter.Attr.cssClass("glyphicon glyphicon-transfer"));
            });
        }));
        return columns;
    }

    private Column column(String title, Consumer<NodeSummary> content, Comparator<NodeSummary> comparator) {
        return new Column(idx -> {
            int column = idx + 1;
            int sort;
            String glyphicon;
            if (column == Math.abs(sortBy)) {
                sort = sortBy * -1;
                if (sort > 0) {
                    glyphicon = "glyphicon glyphicon-sort-by-attributes-alt";
                } else {
                    glyphicon = "glyphicon glyphicon-sort-by-attributes";
                }
            } else {
                glyphicon = null;
                sort = column;
            }

            aHref("/balancer?sortBy=" + sort, () -> {
                if (glyphicon != null) {
                    tag("span", HtmlWriter.Attr.cssClass(glyphicon));
                }
                write(title);
            });
        }, content, comparator);
    }

    private Column column(IntConsumer header, Consumer<NodeSummary> content) {
        return new Column(header, content, Comparator.comparing(NodeSummary::getAddress));
    }

    private static class Column {
        private final IntConsumer header;
        private final Consumer<NodeSummary> content;
        private final Comparator<NodeSummary> comparator;

        public Column(IntConsumer header, Consumer<NodeSummary> content, Comparator<NodeSummary> comparator) {
            this.header = header;
            this.content = content;
            this.comparator = comparator;
        }
    }
}
