package ru.yandex.solomon.balancer.www;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.springframework.http.server.reactive.ServerHttpRequest;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.balancer.Balancer;
import ru.yandex.solomon.balancer.BalancerOptions;
import ru.yandex.solomon.balancer.Resource;
import ru.yandex.solomon.balancer.Resources;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.time.DurationUtils;

import static ru.yandex.solomon.util.time.DurationUtils.formatDurationMillis;

/**
 * @author Vladimir Gordiychuk
 */
public class BalancerSettingsPageWww extends ManagerPageTemplate {
    private final Balancer leader;

    public BalancerSettingsPageWww(Balancer leader) {
        super("Shards balancer settings");
        this.leader = leader;
    }

    @Override
    protected void content() {
        BalancerOptions opts = leader.getOptions();
        BalancerOptions def = getDefaultOptions();
        form(() -> {
            inputGroup(() -> {
                h3("Timeouts");
                table(() -> {
                    rowTime("heartbeatExpiration", "Heartbeat expiration time: ", opts.getHeartbeatExpirationMillis(), def.getHeartbeatExpirationMillis());
                    rowTime("forceUnassignExpiration", "Force unassign timeout: ", opts.getForceUnassignExpirationMillis(), def.getForceUnassignExpirationMillis());
                    rowTime("gracefulUnassignExpiration", "Graceful unassign timeout: ", opts.getGracefulUnassignExpirationMillis(), def.getGracefulUnassignExpirationMillis());
                    rowTime("assignExpiration", "Assign timeout: ", opts.getAssignExpirationMillis(), def.getAssignExpirationMillis());
                });
            });
            inputGroup(() -> {
                h3("Rebalance");
                table(() -> {
                    rowSize("maxReassignInFlight", "Max in-flight shards: ", opts.getMaxReassignInFlight(), def.getMaxReassignInFlight());
                    rowSize("maxLongLoadingShardsToIgnore", "Max long loading shards to ignore: ", opts.getMaxLongLoadingShardsToIgnore(), def.getMaxLongLoadingShardsToIgnore());
                    row("enableAutoRebalance", "Enable auto rebalance: ",
                        String.valueOf(opts.isEnableAutoRebalance()),
                        String.valueOf(def.isEnableAutoRebalance()));
                    row("autoRebalanceDispersionThreshold", "Auto rebalance dispersion threshold: ",
                        String.format("%.5f", opts.getAutoRebalanceDispersionThreshold()),
                        String.format("%.5f", def.getAutoRebalanceDispersionThreshold()));
                    row("rebalanceDispersionTarget", "Rebalance dispersion target: ",
                        String.format("%.5f", opts.getRebalanceDispersionTarget()),
                        String.format("%.5f", def.getRebalanceDispersionTarget()));
                });
            });

            inputGroup(() -> {
                h3("Per node limits");
                Resources res = opts.getLimits();
                Resources defLimit = def.getLimits();
                table(() -> {
                    for (var resource : leader.getResources()) {
                        rowSize(resource.name(), resource.name() + ": ", res.get(resource), defLimit.get(resource));
                    }
                });
            });
            inputGroup(() -> {
                h3("Other");
                table(() -> {
                    row("disableAutoFreeze", "Disable auto freeze: ", String.valueOf(opts.isDisableAutoFreeze()), "false");
                });
            });
            inputHidden("version", Long.toString(opts.getVersion() + 1));
            buttonSubmitDefault("Save");
        }, new Attr("action", "/balancer/settings/save"));
    }

    private BalancerOptions getDefaultOptions() {
        var maximums = new Resources();
        for (var resource : leader.getResources()) {
            double max = resource.maximum(leader.getShards().values(), leader.getNodes().values());
            maximums.set(resource, max);
        }
        return BalancerOptions.newBuilder()
                .setLimits(maximums)
                .build();
    }

    private void rowTime(String id, String name, long millis, long byDefaultMillis) {
        row(id, name, formatDurationMillis(millis), formatDurationMillis(byDefaultMillis));
    }

    private void rowSize(String id, String name, double size, double byDefaultSize) {
        row(id, name, DataSize.shortString(Math.round(size)), DataSize.shortString(Math.round(byDefaultSize)));
    }

    private void row(String id, String name, String value, String byDefault) {
        tr(() -> {
            thText(name);
            td(() -> {
                inputTextfieldFormControl(id, id, value);
            });
            td(() -> {
                tag("span", HtmlWriter.Attr.cssClass("help-block"), () -> {
                    write("(default: " + byDefault + ")");
                });
            });
        });
    }

    public static BalancerOptions parseOptions(ServerHttpRequest r, List<Resource> resources) {
        BalancerOptions.Builder opts = BalancerOptions.newBuilder();

        opts.setHeartbeatExpiration(getTime(r, "heartbeatExpiration"), TimeUnit.MILLISECONDS);
        opts.setForceUnassignExpiration(getTime(r, "forceUnassignExpiration"), TimeUnit.MILLISECONDS);
        opts.setGracefulUnassignExpiration(getTime(r, "gracefulUnassignExpiration"), TimeUnit.MILLISECONDS);
        opts.setAssignExpiration(getTime(r, "assignExpiration"), TimeUnit.MILLISECONDS);
        opts.setMaxReassignInFlight(Math.toIntExact(getSize(r, "maxReassignInFlight")));
        opts.setMaxLongLoadingShardsToIgnore(Math.toIntExact(getSize(r, "maxLongLoadingShardsToIgnore")));
        getValue(r, "enableAutoRebalance").map(Boolean::parseBoolean).ifPresent(opts::setEnableAutoRebalance);
        opts.setAutoRebalanceDispersionThreshold(getDouble(r, "autoRebalanceDispersionThreshold"));
        opts.setRebalanceDispersionTarget(getDouble(r, "rebalanceDispersionTarget"));
        opts.setVersion(getLong(r, "version"));
        getValue(r, "disableAutoFreeze").map(Boolean::parseBoolean).ifPresent(opts::setDisableAutoFreeze);

        Resources limits = new Resources();
        for (var resource : resources) {
            long size = getSize(r, resource.name());
            if (size != 0) {
                limits.set(resource, size);
            }
        }
        opts.setLimits(limits);
        return opts.build();
    }

    private static long getTime(ServerHttpRequest request, String param) {
        return DurationUtils.parseDuration(request.getQueryParams().getFirst(param))
            .orElse(Duration.ZERO)
            .toMillis();
    }

    private static long getSize(ServerHttpRequest request, String param) {
        String value = request.getQueryParams().getFirst(param);
        if (value == null || "".equals(value)) {
            return 0;
        }

        if (StringUtils.isNumeric(value)) {
            return Long.parseLong(value);
        }

        return DataSize.valueOf(value).toBytes();
    }

    private static Optional<String> getValue(ServerHttpRequest request, String param) {
        String value = request.getQueryParams().getFirst(param);
        if (value == null || "".equals(value)) {
            return Optional.empty();
        }

        return Optional.of(value);
    }

    private static long getLong(ServerHttpRequest request, String param) {
        String value = request.getQueryParams().getFirst(param);
        if (value == null || "".equals(value)) {
            return 0;
        }

        return Long.parseLong(value);
    }

    private static double getDouble(ServerHttpRequest request, String param) {
        String value = request.getQueryParams().getFirst(param);
        if (value == null || "".equals(value)) {
            return 0.0;
        }

        return Double.parseDouble(value);
    }
}
