package ru.yandex.solomon.balancer;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import io.grpc.Status;
import org.hamcrest.Matchers;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.balancer.dao.InMemoryBalancerDao;
import ru.yandex.solomon.balancer.remote.RemoteClusterStub;
import ru.yandex.solomon.balancer.remote.RemoteNode;
import ru.yandex.solomon.balancer.remote.RemoteNodeState;
import ru.yandex.solomon.balancer.remote.RemoteShardState;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static org.hamcrest.CoreMatchers.hasItem;
import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.balancer.CommonResource.CPU;
import static ru.yandex.solomon.balancer.CommonResource.MEMORY;
import static ru.yandex.solomon.balancer.CommonResource.SHARDS_COUNT;

/**
 * @author Vladimir Gordiychuk
 */
public class BalancerImplTest {
    private static final Logger logger = LoggerFactory.getLogger(BalancerImplTest.class);
    private static final String[] NONE_SHARDS = new String[0];

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(10, TimeUnit.SECONDS)
            .build();

    private ManualClock clock;
    private ManualScheduledExecutorService executorService;
    private RemoteClusterStub cluster;
    private ShardsHolderStub shardsHolder;
    private TotalShardCounter shardCounter;
    private InMemoryBalancerDao dao;
    private BalancerImpl leader;
    private TestRemoteNode alice;
    private TestRemoteNode bob;
    private TestRemoteNode eva;

    @Before
    public void setUp() {
        clock = new ManualClock();
        executorService = new ManualScheduledExecutorService(2, clock);

        cluster = new RemoteClusterStub();
        shardsHolder = new ShardsHolderStub();
        shardsHolder.syncAdd(IntStream.rangeClosed(1, 128)
                .mapToObj(Integer::toString)
                .toArray(String[]::new));
        shardCounter = new TotalShardCounterImpl();
        dao = new InMemoryBalancerDao();
        leader = createLeader(42);

        var limits = new Resources();
        limits.set(SHARDS_COUNT, 1024);
        leader.setOptions(leader.getOptions()
            .toBuilder()
            .setHeartbeatExpiration(15, TimeUnit.SECONDS)
            .setForceUnassignExpiration(1, TimeUnit.SECONDS)
            .setGracefulUnassignExpiration(15, TimeUnit.SECONDS)
            .setAssignExpiration(1, TimeUnit.SECONDS)
            .setMaxReassignInFlight(shardsHolder.getShards().size() / 2)
            .setMaxLongLoadingShardsToIgnore(1)
            .setDisableAutoFreeze(true)
            .setLimits(limits)
            .build())
            .join();

        alice = new TestRemoteNode("alice");
        bob = new TestRemoteNode("bob");
        eva = new TestRemoteNode("eva");
        clock.passedTime(10, TimeUnit.MINUTES);
    }

    @After
    public void tearDown() {
        leader.close();
        executorService.shutdownNow();
    }

    @Test
    public void assignShardToNode() throws InterruptedException {
        assertArrayEquals(NONE_SHARDS, alice.getAssignedShards());

        registerNodes(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);

        var assigned = alice.getAssignedShards();
        assertArrayEquals(allShards(), assigned);
    }

    @Test
    public void assignOnlyWhenAllNodeVisible() throws InterruptedException {
        cluster.register(alice, bob);
        awaitAct();

        awaitUpdateState(alice);
        assertFalse(onMessageSync(alice, bob).await(5, TimeUnit.MILLISECONDS));

        // no assignments, because not all nodes in the cluster respond on ping to leader
        assertArrayEquals(NONE_SHARDS, alice.getAssignedShards());
        assertArrayEquals(NONE_SHARDS, bob.getAssignedShards());

        // every node in the cluster responded on ping to leader, and now leader can assign shards by nodes
        awaitUpdateState(alice, bob);
        awaitAllShardAssigned(alice, bob);

        // all shards spread by all live nodes in the cluster
        assertNotEquals(0, alice.getAssignedShards().length);
        assertNotEquals(0, bob.getAssignedShards().length);

        // one shard can not be assigned on more then one node
        var assigned = getAssignedShards(alice, bob);
        assertArrayEquals(allShards(), assigned);
    }

    @Test
    public void assignOnlyWhenAllNodeVisibleOrExpired() throws InterruptedException {
        cluster.register(alice, bob, eva);

        // only alice currently alive, after expire expire bob heartbeat should be started assignment
        CountDownLatch sync = onMessageSync(alice);
        for (int index = 0; index < 5; index++) {
            alice.updateRemoteState();
            awaitAct();
            long delay = leader.getOptions().getHeartbeatExpirationMillis();
            assertTrue(delay > 0);
            clock.passedTime(delay / 3, TimeUnit.MILLISECONDS);
            if (sync.await(10, TimeUnit.MILLISECONDS)) {
                break;
            }
        }

        awaitAllShardAssigned(alice);

        var assigned = alice.getAssignedShards();
        assertArrayEquals(allShards(), assigned);
    }

    @Test
    public void avoidMoveShardsWhenNewLeaderAcquiredLock() throws InterruptedException {
        assignAllShards(alice, bob, eva);
        var assignedAlice = alice.getAssignedShards();
        var assignedBob = bob.getAssignedShards();
        var assignedEva = eva.getAssignedShards();

        cluster.register(alice, bob, eva);
        updateRemoteState(alice, bob, eva);

        for (int index = 0; index < 10; index++) {
            updateRemoteState(alice, bob, eva);
        }

        assertArrayEquals(assignedAlice, alice.getAssignedShards());
        assertArrayEquals(assignedBob, bob.getAssignedShards());
        assertArrayEquals(assignedEva, eva.getAssignedShards());
    }

    @Test
    public void reassignShardsWhenHeartbeatExpired() throws InterruptedException {
        assignAllShards(alice, bob);

        cluster.register(alice, bob);
        updateRemoteState(alice, bob);

        // latest heartbeat from bob expired
        CountDownLatch sync = onMessageSync(alice);
        for (int index = 0; index < 30; index++) {
            updateRemoteState(alice);
            clock.passedTime(3, TimeUnit.SECONDS);
            if (sync.await(10, TimeUnit.MILLISECONDS)) {
                break;
            }
        }

        // now all assignments from bob should be reassign to alice
        awaitAllShardAssigned(alice);

        assertArrayEquals(allShards(), alice.getAssignedShards());

        // bob also receive force unassign commands
        awaitAllShardUnassigned(bob);
        assertArrayEquals(NONE_SHARDS, bob.getAssignedShards());
    }

    @Test
    public void removeExpiredAndExcludedNode() throws InterruptedException {
        assignAllShards(alice, bob);

        cluster.register(alice, bob);
        updateRemoteState(alice, bob);

        updateRemoteState(alice, bob);
        cluster.unregister(bob);
        do {
            updateRemoteState(alice);
            clock.passedTime(leader.getOptions().getHeartbeatExpirationMillis() / 3, TimeUnit.MILLISECONDS);
        } while (cluster.getNodes().contains(bob.name));

        // now all assignments from bob should be reassign to alice
        awaitAllShardAssigned(alice);
        assertArrayEquals(allShards(), alice.getAssignedShards());

        var nodes = leader.getNodes();
        assertEquals(Set.of(alice.name), nodes.keySet());
    }

    @Test
    public void assignSameShardsOnRestart() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);

        clock.passedTime(5, TimeUnit.SECONDS);
        updateRemoteState(10, alice, bob);
        clock.passedTime(10, TimeUnit.SECONDS);

        var prevAssignedBob = bob.getAssignedShards();
        logger.info("bob assignments: {}", Arrays.toString(prevAssignedBob));
        logger.info("bob process failed, and restart fast enough");
        bob.reset();
        cluster.register(bob);

        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);

        assertArrayEquals(prevAssignedBob, bob.getAssignedShards());
    }

    @Test
    public void failedAssignments() throws InterruptedException {
        eva.assignStatus = Status.INTERNAL
            .withDescription("eva broken, but still send heartbeats");

        cluster.register(alice, bob, eva);
        for (int index = 0; index < 10; index++) {
            updateRemoteState(alice, bob, eva);
            eva.failedCommands = index * 10;
            clock.passedTime(1, TimeUnit.SECONDS);
        }

        awaitAllShardAssigned(alice, bob, eva);
        assertArrayEquals("eva can't accept assignments, because broken", NONE_SHARDS, eva.getAssignedShards());

        var assigned = getAssignedShards(alice, bob);
        assertArrayEquals(allShards(), assigned);
    }

    @Test
    public void autoRebalance() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);
        leader.setOptions(leader.getOptions()
            .toBuilder()
            .setEnableAutoRebalance(true)
            .setAutoRebalanceDispersionThreshold(0.8)
            .build())
            .join();

        clock.passedTime(1, TimeUnit.SECONDS);
        logger.info("Adding new nodes to cluster, shards should be rebalanced");
        cluster.register(bob, eva);

        while (leader.getRebalanceProgress() < 1d || eva.getAssignedShards().length == 0) {
            awaitAllShardAssigned(alice, bob, eva);
            updateRemoteState(1, alice, bob, eva);
        }
        awaitAllShardAssigned(alice, bob, eva);

        double dispersion = leader.getDispersion();
        logger.info("Dispersion: {}", dispersion);
        assertEquals(0, dispersion, 0.52);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
        assertNotEquals(0, alice.getAssignedShards().length);
        assertNotEquals(0, bob.getAssignedShards().length);
        assertNotEquals(0, eva.getAssignedShards().length);
    }

    @Test
    public void autoRebalanceFromSeveralNodes() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);
        leader.setOptions(leader.getOptions()
            .toBuilder()
            .setEnableAutoRebalance(true)
            .setMaxReassignInFlight(3)
            .setAutoRebalanceDispersionThreshold(0.8)
            .setRebalanceDispersionTarget(0.001)
            .build())
            .join();

        clock.passedTime(1, TimeUnit.SECONDS);
        logger.info("Adding new node to cluster, shards should be rebalanced");
        cluster.register(eva);

        while (leader.getRebalanceProgress() < 1d || eva.getAssignedShards().length == 0) {
            awaitAllShardAssigned(alice, bob, eva);
            updateRemoteState(1, alice, bob, eva);
        }
        awaitAllShardAssigned(alice, bob, eva);

        double dispersion = leader.getDispersion();
        logger.info("Dispersion: {}", dispersion);
        assertEquals(0, dispersion, 0.52);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
        assertNotEquals(0, alice.getAssignedShards().length);
        assertNotEquals(0, bob.getAssignedShards().length);
        assertNotEquals(0, eva.getAssignedShards().length);
    }

    @Test
    public void autoRebalanceWithTarget() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);
        leader.setOptions(leader.getOptions()
            .toBuilder()
            .setMaxReassignInFlight(3)
            .setEnableAutoRebalance(true)
            .setAutoRebalanceDispersionThreshold(0.9)
            .setRebalanceDispersionTarget(0.7)
            .build())
            .join();

        clock.passedTime(1, TimeUnit.SECONDS);
        logger.info("Adding new nodes to cluster, shards should be rebalanced");
        cluster.register(bob, eva);

        while (leader.getRebalanceProgress() < 1d || eva.getAssignedShards().length == 0) {
            awaitAllShardAssigned(alice, bob, eva);
            updateRemoteState(1, alice, bob, eva);
        }
        awaitAllShardAssigned(alice, bob, eva);

        double dispersion = leader.getDispersion();
        logger.info("Dispersion: {}", dispersion);
        assertEquals(dispersion, 0.7, 0.05);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
        assertNotEquals(0, alice.getAssignedShards().length);
        assertNotEquals(0, bob.getAssignedShards().length);
        assertNotEquals(0, eva.getAssignedShards().length);
    }

    @Test
    public void autoRebalanceDisabled() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);

        leader.setOptions(
                leader.getOptions().toBuilder()
                    .setEnableAutoRebalance(false)
                    .setAutoRebalanceDispersionThreshold(0.0001)
                    .build())
            .join();

        clock.passedTime(1, TimeUnit.SECONDS);
        assertEquals(0, leader.getDispersion(), 0.5);

        logger.info("Adding new node to cluster, but auto rebalance is not enabled");
        cluster.register(eva);

        updateRemoteState(1, alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);
        awaitAct();
        assertEquals(1, leader.getDispersion(), 0);

        var aliceAssignment = alice.getAssignedShards();
        var bobAssignment = bob.getAssignedShards();

        for (int index = 0; index < 100; index++) {
            updateRemoteState(alice, bob, eva);
            awaitAllShardAssigned(alice, bob, eva);

            assertEquals(1, leader.getRebalanceProgress(), 0);
            assertArrayEquals(aliceAssignment, alice.getAssignedShards());
            assertArrayEquals(bobAssignment, bob.getAssignedShards());
            assertArrayEquals(NONE_SHARDS, eva.getAssignedShards());
        }
    }

    @Test
    public void autoRebalanceDontStartWhenDispersionUnderThreshold() throws InterruptedException {
        cluster.register(alice, bob, eva);
        updateRemoteState(alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);

        leader.setOptions(leader.getOptions()
            .toBuilder()
            .setEnableAutoRebalance(true)
            .setAutoRebalanceDispersionThreshold(0.8)
            .build()).join();

        clock.passedTime(1, TimeUnit.SECONDS);
        assertEquals(0, leader.getDispersion(), 0.5);

        leader.setActive(eva.name, false).join();
        leader.kickNode(eva.name).join();

        awaitAllShardAssigned(alice, bob, eva);
        assertEquals(0, leader.getDispersion(), 0.5);

        var aliceAssignment = alice.getAssignedShards();
        var bobAssignment = bob.getAssignedShards();

        for (int index = 0; index < 100; index++) {
            updateRemoteState(alice, bob, eva);
            awaitAllShardAssigned(alice, bob, eva);

            assertEquals(1, leader.getRebalanceProgress(), 0);
            assertArrayEquals(aliceAssignment, alice.getAssignedShards());
            assertArrayEquals(bobAssignment, bob.getAssignedShards());
        }
    }

    @Test
    public void correctResourceUsageCalculation() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(3, alice);
        awaitAllShardAssigned(alice);
        checkResourceUsage();

        cluster.register(bob, eva);
        updateRemoteState(3, alice, bob, eva);
        checkResourceUsage();
        awaitRebalance(alice, bob, eva);
        checkResourceUsage();

        leader.setActive(bob.getName(), false).join();
        awaitRebalance(alice, bob, eva);
        checkResourceUsage();
    }

    private void checkResourceUsage() throws InterruptedException {
        // ensure that all heartbeats processed
        awaitAct();
        awaitAct();

        for (NodeSummary node : leader.getNodes().values()) {
            var nodeUsage = node.getResources();
            var shardsSumUsage = node.getShards()
                .values()
                .stream()
                .map(ShardSummary::getResources)
                .reduce(new Resources(), Resources::combine);

            for (var resource : shardsSumUsage.keys()) {
                double expected = shardsSumUsage.get(resource);
                double actual = nodeUsage.get(resource);
                assertEquals("Node "+ node.getAddress() + " usage of resource " + resource, expected, actual, 0.1);
            }
        }
    }

    @Test
    public void kickShard() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);

        logger.info("Live node into cluster grow");
        cluster.register(alice, bob);

        updateRemoteState(alice, bob);

        leader.kickShard("1").join();
        awaitAllShardAssigned(alice, bob);

        var expectedAlice = Stream.of(allShards())
            .filter(shardId -> !shardId.equals("1"))
            .toArray(String[]::new);

        assertArrayEquals(new String[]{"1"}, bob.getAssignedShards());
        assertArrayEquals(expectedAlice, alice.getAssignedShards());
    }

    @Test
    public void kickNode() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);

        logger.info("Live node into cluster grow");
        cluster.register(alice, bob);

        updateRemoteState(alice, bob);

        leader.kickNode(alice.getName()).join();
        awaitAllShardAssigned(alice, bob);

        assertNotEquals(0, bob.getAssignedShards().length);
        assertNotEquals(0, alice.getAssignedShards().length);

        var assigned = getAssignedShards(alice, bob);
        assertArrayEquals(allShards(), assigned);
    }

    @Test
    public void avoidAssignToInactiveNode() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);

        logger.info("Live node into cluster grow");
        cluster.register(alice, bob);

        updateRemoteState(alice, bob);
        leader.setActive(alice.getName(), false).join();
        leader.kickNode(alice.getName()).join();
        awaitAllShardAssigned(alice, bob);

        assertArrayEquals(NONE_SHARDS, alice.getAssignedShards());
        assertArrayEquals(allShards(), bob.getAssignedShards());
    }

    @Test
    public void bulkSetActive() throws InterruptedException {
        cluster.register(alice, bob, eva);
        updateRemoteState(alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);

        updateRemoteState(alice, bob, eva);
        leader.setActive(false).join();
        leader.setActive(bob.getName(), true).join();
        leader.kickNode(alice.getName()).join();
        leader.kickNode(eva.getName()).join();

        awaitAllShardAssigned(alice, bob, eva);

        assertArrayEquals(NONE_SHARDS, alice.getAssignedShards());
        assertArrayEquals(NONE_SHARDS, eva.getAssignedShards());
        assertArrayEquals(allShards(), bob.getAssignedShards());

        leader.setActive(true).join();
        leader.kickNode(bob.getName()).join();

        awaitAllShardAssigned(alice, bob, eva);
        assertNotEquals(0, alice.getAssignedShards().length);
        assertNotEquals(0, bob.getAssignedShards().length);
        assertNotEquals(0, eva.getAssignedShards().length);

        var assigned = getAssignedShards(alice, bob, eva);
        assertArrayEquals(allShards(), assigned);
    }

    @Test
    public void kickFreezeNodeUnavailable() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);

        leader.setFreeze(alice.getName(), true).join();
        cluster.register(bob);
        updateRemoteState(alice, bob);

        assertArrayEquals(allShards(), alice.getAssignedShards());

        leader.kickNode(alice.getName()).join();
        awaitAllShardAssigned(alice, bob);

        assertArrayEquals(allShards(), alice.getAssignedShards());
        assertArrayEquals(NONE_SHARDS, bob.getAssignedShards());

        leader.setFreeze(alice.getName(), false).join();
        leader.kickNode(alice.getName()).join();
        awaitAllShardAssigned(alice, bob);

        assertNotEquals(0, alice.getAssignedShards().length);
        assertNotEquals(0, bob.getAssignedShards().length);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob));
    }

    @Test
    public void freezeRemindAssignments() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob));

        var bobAssignments = bob.getAssignedShards();

        logger.info("ensure that after restart bob process assignment will be same because node freeze");
        leader.setFreeze(bob.name, true).join();
        logger.info("bob process failed");
        bob.reset();
        clock.passedTime(15, TimeUnit.SECONDS);
        updateRemoteState(3, alice, bob);
        awaitAllShardAssigned(alice, bob);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob));
        assertArrayEquals(bobAssignments, bob.getAssignedShards());
    }

    @Test
    public void freezeAvoidUnassignExpired() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob));

        var bobAssignments = bob.getAssignedShards();

        leader.setFreeze(bob.name, true).join();

        // latest heartbeat from bob expired
        do {
            updateRemoteState(1, alice);
            clock.passedTime(leader.getOptions().getAssignExpirationMillis() / 3, TimeUnit.MILLISECONDS);
        } while (leader.getNodes().get(bob.getName()).getStatus() != NodeStatus.EXPIRED);
        awaitAllShardAssigned(alice, bob);

        updateRemoteState(3, alice, bob);
        awaitAllShardAssigned(alice, bob);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob));
        assertArrayEquals(bobAssignments, bob.getAssignedShards());
    }

    @Test
    public void frezeAvoidUnassignFreezeOnRebalance() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob));

        var bobAssignments = bob.getAssignedShards();

        leader.setFreeze(bob.name, true).join();

        cluster.register(eva);
        updateRemoteState(3, alice, bob, eva);
        awaitRebalance(alice, bob, eva);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
        assertArrayEquals(bobAssignments, bob.getAssignedShards());
    }

    @Test
    public void freezePersistence2() throws InterruptedException {
        cluster.register(alice, bob, eva);
        updateRemoteState(alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));

        var bobAssignments = bob.getAssignedShards();
        var aliceAssignments = alice.getAssignedShards();

        leader.setFreeze(eva.name, true).join();
        leader.setFreeze(bob.name, true).join();
        leader.setFreeze(alice.name, true).join();

        clock.passedTime(5, TimeUnit.SECONDS);
        updateRemoteState(3, alice, bob, eva);

        leader = createLeader(43);
        do {
            updateRemoteState(1, eva);
            awaitAllShardAssigned(alice, bob, eva);
            clock.passedTime(5, TimeUnit.SECONDS);
        } while (leader.getNodes().get(alice.getName()).getStatus() != NodeStatus.EXPIRED);

        bob.reset();
        bob.assignStatus = Status.UNAVAILABLE.withDescription("bob broken, and fail again and again");

        updateRemoteState(3, alice, bob, eva);
        bob.assignStatus = Status.OK;
        updateRemoteState(3, alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);

        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
        assertArrayEquals(bobAssignments, bob.getAssignedShards());
        assertArrayEquals(aliceAssignments, alice.getAssignedShards());

        awaitRebalance(alice, bob, eva);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
        assertArrayEquals(bobAssignments, bob.getAssignedShards());
        assertArrayEquals(aliceAssignments, alice.getAssignedShards());
    }

    @Test
    public void shardStayOnFreezeNodeAfterHeartbeatExpire() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);

        var aliceAssign = alice.getAssignedShards();
        var bobAssign = bob.getAssignedShards();
        leader.setFreeze(alice.getName(), true).join();
        // latest heartbeat from alice expired
        do {
            updateRemoteState(bob);
            clock.passedTime(leader.getOptions().getHeartbeatExpirationMillis() / 3, TimeUnit.MILLISECONDS);
        } while (leader.getNodes().get(alice.getName()).getStatus() != NodeStatus.EXPIRED);

        awaitAllShardAssigned(alice, bob);
        assertArrayEquals(aliceAssign, alice.getAssignedShards());
        assertArrayEquals(bobAssign, bob.getAssignedShards());

        leader.setFreeze(alice.getName(), false).join();
        awaitAllShardAssigned(bob);
        assertArrayEquals(allShards(), bob.getAssignedShards());
        assertArrayEquals(NONE_SHARDS, alice.getAssignedShards());
    }

    @Test
    public void notAbleAssignOnFreezeNode() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);

        cluster.register(bob);
        updateRemoteState(alice, bob);
        leader.setFreeze(bob.getName(), true).join();
        leader.kickNode(alice.name).join();
        awaitUpdateState(alice, bob);

        assertArrayEquals(allShards(), alice.getAssignedShards());
        assertArrayEquals(NONE_SHARDS, bob.getAssignedShards());

        leader.setFreeze(bob.getName(), false).join();
        updateRemoteState(alice, bob);
        leader.kickNode(alice.getName()).join();
        awaitAllShardAssigned(alice, bob);

        assertNotEquals(0, alice.getAssignedShards().length);
        assertNotEquals(0, bob.getAssignedShards().length);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob));
    }

    @Test
    public void onRestartFreezeNodeGetSameAssignments() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);

        var aliceAssign = alice.getAssignedShards();
        var bobAssign = bob.getAssignedShards();
        leader.setFreeze(alice.getName(), true).join();
        // latest heartbeat from alice expired
        do {
            updateRemoteState(bob);

            clock.passedTime(3, TimeUnit.SECONDS);
            TimeUnit.MILLISECONDS.sleep(10);
        } while (leader.getNodes().get(alice.getName()).getStatus() != NodeStatus.EXPIRED);
        awaitAllShardAssigned(alice, bob);

        logger.debug("alice process restart now");
        alice.reset();
        updateRemoteState(alice, bob);

        awaitAllShardAssigned(alice, bob);
        assertArrayEquals(aliceAssign, alice.getAssignedShards());
        assertArrayEquals(bobAssign, bob.getAssignedShards());
    }

    @Test
    public void shardAssignmentPersistent() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);

        var aliceAssign = alice.getAssignedShards();
        var bobAssign = bob.getAssignedShards();
        logger.info("full cluster restart");
        awaitAct();

        leader.close();
        alice.reset();
        bob.reset();
        eva.reset();
        cluster.register(alice, bob, eva);

        logger.info("new leader elected");
        leader = createLeader(43);

        awaitAct();
        updateRemoteState(2, alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);

        assertArrayEquals(aliceAssign, alice.getAssignedShards());
        assertArrayEquals(bobAssign, bob.getAssignedShards());
        assertArrayEquals(NONE_SHARDS, eva.getAssignedShards());
        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
    }

    @Test
    public void freezePersistent() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);

        assertArrayEquals(allShards(), getAssignedShards(alice, bob));
        leader.setFreeze(bob.getName(), true).join();

        var bobAssign = bob.getAssignedShards();
        awaitAct();
        logger.info("full cluster restart");

        leader.close();
        alice.reset();
        bob.reset();
        eva.reset();

        logger.info("new leader elected");
        leader = createLeader(44);

        awaitAct();
        updateRemoteState(2, alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);

        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
        assertArrayEquals(bobAssign, bob.getAssignedShards());

        assertTrue(leader.getNodes().get(bob.getName()).isFreeze());
        leader.kickNode(bob.getName()).join();
        awaitAllShardAssigned(alice, bob, eva);

        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
        assertArrayEquals(bobAssign, bob.getAssignedShards());
    }

    @Test
    public void optsPersistent() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);

        awaitAct();

        logger.info("restart leader");
        leader.setOptions(leader.getOptions()
            .toBuilder()
            .setVersion(4200)
            .build())
            .join();

        awaitAct();
        leader.close();

        leader = createLeader(45);
        for (int index = 0; index < 3; index++) {
            updateRemoteState(alice, bob);
            awaitAct();
        }
        awaitAllShardAssigned(alice, bob);
        assertArrayEquals(allShards(), getAssignedShards(alice, bob));
        assertEquals(4200, leader.getOptions().getVersion());
    }

    @Test
    public void rebalance() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);

        var aliceAssignment = alice.getAssignedShards();
        var bobAssignment = bob.getAssignedShards();
        assertArrayEquals(allShards(), getAssignedShards(alice, bob));

        updateRemoteState(3, alice, bob);
        double initDispersion = leader.getDispersion();
        logger.info("Dispersion of two node: {}", initDispersion);
        assertEquals(0, initDispersion, 0.7);

        cluster.register(eva);
        updateRemoteState(3, alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);
        assertArrayEquals(NONE_SHARDS, eva.getAssignedShards());
        double badDispersion = leader.getDispersion();
        logger.info("Init dispersion: {}", initDispersion);
        logger.info("Dispersion after add new node: {}", badDispersion);
        assertTrue(badDispersion > initDispersion);
        awaitRebalance(alice, bob, eva);

        logger.info("Dispersion: {}", leader.getDispersion());
        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
        assertNotEquals(aliceAssignment.length, alice.getAssignedShards().length);
        assertNotEquals(bobAssignment.length, bob.getAssignedShards().length);
        assertNotEquals(0, eva.getAssignedShards().length);
        assertTrue(leader.getDispersion() < badDispersion);
    }

    @Test
    public void rebalanceSkipFreezeShards() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);
        updateRemoteState(alice, bob);

        assertArrayEquals(allShards(), getAssignedShards(alice, bob));
        leader.setFreeze(bob.name, true).join();

        var bobAssignment = bob.getAssignedShards();

        cluster.register(eva);
        updateRemoteState(3, alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);
        awaitRebalance(alice, bob, eva);

        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
        assertArrayEquals(bobAssignment, bob.getAssignedShards());
        assertNotEquals(0, eva.getAssignedShards().length);
    }

    @Test
    public void rebalanceMoveShardsFromInactiveNode() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);
        updateRemoteState(alice, bob);

        assertArrayEquals(allShards(), getAssignedShards(alice, bob));

        leader.setActive(bob.name, false).join();
        cluster.register(eva);
        updateRemoteState(3, alice, bob, eva);
        awaitRebalance(alice, bob, eva);

        assertArrayEquals(allShards(), getAssignedShards(alice, bob, eva));
        assertArrayEquals(NONE_SHARDS, bob.getAssignedShards());
        assertNotEquals(0, eva.getAssignedShards().length);
    }

    @Test
    public void increaseAmountOfShards() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);
        updateRemoteState(alice, bob);

        assertArrayEquals(allShards(), getAssignedShards(alice, bob));

        int count = shardsHolder.getShards().size();
        int expectedShardsCount = count + 5;
        for (int index = count + 1; index <= expectedShardsCount; index++) {
            shardsHolder.syncAdd(Integer.toString(index));
        }
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);

        var assignments = getAssignedShards(alice, bob);
        assertEquals(expectedShardsCount, assignments.length);
        assertArrayEquals(allShards(), assignments);
    }

    @Test
    public void fullClusterRestartGetSameAssignments() throws InterruptedException {
        leader.setOptions(leader.getOptions()
                .toBuilder()
                .setDisableAutoFreeze(false)
                .build())
                .join();

        cluster.register(alice, bob, eva);
        updateRemoteState(alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);

        var aliceAssign = alice.getAssignedShards();
        var bobAssign = bob.getAssignedShards();
        var evaAssign = eva.getAssignedShards();

        alice.reset();
        bob.reset();
        eva.reset();

        updateRemoteState(alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);

        assertArrayEquals(aliceAssign, alice.getAssignedShards());
        assertArrayEquals(bobAssign, bob.getAssignedShards());
        assertArrayEquals(evaAssign, eva.getAssignedShards());
    }

    @Test
    public void fullClusterRestartNoMoveFirstTenMin() throws InterruptedException {
        leader.setOptions(leader.getOptions()
                .toBuilder()
                .setDisableAutoFreeze(false)
                .build())
                .join();

        cluster.register(alice, bob, eva);
        updateRemoteState(alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);

        var aliceAssign = alice.getAssignedShards();
        var bobAssign = bob.getAssignedShards();
        var evaAssign = eva.getAssignedShards();

        alice.reset();
        bob.reset();
        eva.reset();
        // after full cluster restart eva not
        do {
            updateRemoteState(alice, bob);

            clock.passedTime(3, TimeUnit.SECONDS);
            TimeUnit.MILLISECONDS.sleep(10);
        } while (leader.getNodes().get(eva.getName()).getStatus() != NodeStatus.EXPIRED);

        assertArrayEquals(aliceAssign, alice.getAssignedShards());
        assertArrayEquals(bobAssign, bob.getAssignedShards());

        // eva alive now
        updateRemoteState(3, alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);

        assertArrayEquals(aliceAssign, alice.getAssignedShards());
        assertArrayEquals(bobAssign, bob.getAssignedShards());
        assertArrayEquals(evaAssign, eva.getAssignedShards());
    }

    @Test
    public void avoidReasignWhenHalfClusterUnavailable() throws InterruptedException {
        leader.setOptions(leader.getOptions()
                .toBuilder()
                .setDisableAutoFreeze(false)
                .build())
                .join();

        cluster.register(alice, bob, eva);
        updateRemoteState(alice, bob, eva);
        awaitAllShardAssigned(alice, bob, eva);

        var aliceAssign = alice.getAssignedShards();
        var bobAssign = bob.getAssignedShards();
        var evaAssign = eva.getAssignedShards();
        // only one node from 3 available, better not to do anything
        do {
            updateRemoteState(alice);

            clock.passedTime(3, TimeUnit.SECONDS);
            TimeUnit.MILLISECONDS.sleep(10);
        } while (nodeStatus(eva) != NodeStatus.EXPIRED || nodeStatus(bob) != NodeStatus.EXPIRED);

        awaitAllShardAssigned(alice, bob, eva);
        assertArrayEquals(aliceAssign, alice.getAssignedShards());
        assertArrayEquals(bobAssign, bob.getAssignedShards());
        assertArrayEquals(evaAssign, eva.getAssignedShards());

        // all nodes now available
        updateRemoteState(alice, bob, eva);
        assertArrayEquals(aliceAssign, alice.getAssignedShards());
        assertArrayEquals(bobAssign, bob.getAssignedShards());
        assertArrayEquals(evaAssign, eva.getAssignedShards());
    }

    @Test
    public void assignIncrementSeqNo() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);

        String shardId = allShards()[0];

        var fist = alice.assignments.get(shardId);
        assertNotEquals(0, fist.getLeaderSeqNo());
        assertNotEquals(0, fist.getAssignSeqNo());

        // latest heartbeat from alice expired
        cluster.register(bob);
        CountDownLatch sync = onMessageSync(bob);
        do {
            updateRemoteState(bob);

            clock.passedTime(3, TimeUnit.SECONDS);
            TimeUnit.MILLISECONDS.sleep(10);
        } while (nodeStatus(alice) != NodeStatus.EXPIRED);

        // now all assignments from alice should be reassign to bob
        awaitAllShardAssigned(bob);

        var second = bob.assignments.get(shardId);
        assertEquals(fist.getLeaderSeqNo(), second.getLeaderSeqNo());
        assertThat(fist, Matchers.lessThan(second));
    }

    @Test
    public void avoidAssignEmptyShard() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);

        String host = leader.getAssignment("myProject").join();

        assertNull(host);
        awaitAllShardAssigned(alice);
        assertArrayEquals(allShards(), alice.getAssignedShards());
    }

    @Test
    public void createShardForExistProject() throws InterruptedException {
        final var projectId = "myNewProject";
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);

        String host = leader.getOrCreateAssignment(projectId).join();
        assertEquals(alice.name, host);

        var expected = allShards();
        assertThat(Set.of(alice.getAssignedShards()), hasItem(projectId));
        assertArrayEquals(expected, alice.getAssignedShards());
    }

    @Test
    public void createShardForExistShardAfterReload() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);
        awaitAct();

        leader = createLeader(43);
        awaitAct();

        var shardId = allShards()[0];
        var host = leader.getOrCreateAssignment(shardId).join();
        assertEquals(alice.name, host);
    }

    @Test
    public void deleteShardOnDeleteProject() throws InterruptedException {
        cluster.register(alice);
        updateRemoteState(alice);
        awaitAllShardAssigned(alice);
        awaitAct();

        String deleteShard = allShards()[0];
        shardsHolder.shards.remove(deleteShard);
        do {
            awaitAct();
        } while (leader.getShards().containsKey(deleteShard) || Set.of(alice.getAssignedShards()).contains(deleteShard));

        var expectedShards = new HashSet<>(Arrays.asList(allShards()));
        expectedShards.remove(deleteShard);
        assertEquals(expectedShards, Set.of(alice.getAssignedShards()));
    }

    @Test
    public void unassignNotActualShardsFromNode() throws InterruptedException {
        cluster.register(alice, bob);
        updateRemoteState(alice, bob);
        awaitAllShardAssigned(alice, bob);
        awaitAct();

        var copyAliceAssignments = new HashMap<>(alice.assignments);

        // latest heartbeat from alice expired
        CountDownLatch sync = onMessageSync(bob);
        for (int index = 0; index < 30; index++) {
            updateRemoteState(bob);
            clock.passedTime(3, TimeUnit.SECONDS);
            if (sync.await(10, TimeUnit.MILLISECONDS)) {
                break;
            }
        }

        // now all assignments from alice should be reassign to bob
        awaitAllShardAssigned(bob);

        // alice live again
        assertEquals(Map.of(), alice.assignments);
        alice.assignments.putAll(copyAliceAssignments);

        while (!alice.assignments.isEmpty()) {
            updateRemoteState(alice, bob);
        }

        awaitAllShardAssigned(alice, bob);
        assertArrayEquals(NONE_SHARDS, alice.getAssignedShards());
    }

    @Test
    public void shardCounterUpdated() throws Exception {
        int initialCount = shardsHolder.getShards().size();

        awaitAct();
        awaitAct();
        assertEquals(initialCount, shardCounter.getTotalShardCount());

        int updatedCount = initialCount + 5;
        for (int index = initialCount + 1; index <= updatedCount; index++) {
            shardsHolder.syncAdd(Integer.toString(index));
        }

        awaitAct();
        awaitAct();
        assertEquals(updatedCount, shardCounter.getTotalShardCount());
    }

    private NodeStatus nodeStatus(TestRemoteNode node) {
        return leader.getNodes().get(node.getName()).getStatus();
    }

    private String[] allShards() {
        return shardsHolder.getShards()
                .stream()
                .sorted()
                .toArray(String[]::new);
    }

    private void awaitRebalance(TestRemoteNode... nodes) throws InterruptedException {
        logger.info("manual trigger rebalance process");
        leader.rebalance().join();
        while (leader.getRebalanceProgress() < 1d) {
            awaitAllShardAssigned(nodes);
            updateRemoteState(nodes);
        }
        awaitAllShardAssigned(nodes);
        logger.info("rebalance process done");
    }

    private void awaitAct() throws InterruptedException {
        CountDownLatch sync = new CountDownLatch(1);
        leader.run(sync::countDown);
        sync.await();
    }

    private String[] getAssignedShards(TestRemoteNode... nodes) {
        return Stream.of(nodes)
            .flatMap(node -> Stream.of(node.getAssignedShards()))
            .sorted()
            .toArray(String[]::new);
    }

    private void updateRemoteState(int count, TestRemoteNode... nodes) throws InterruptedException {
        for (int index = 0; index < count; index++) {
            for (var node : nodes) {
                node.updateRemoteState();
            }
            awaitAct();
        }
    }

    private void updateRemoteState(TestRemoteNode... nodes) throws InterruptedException {
        updateRemoteState(1, nodes);
    }

    private CountDownLatch onMessageSync(TestRemoteNode... nodes) {
        CountDownLatch sync = new CountDownLatch(1);
        CompletableFuture.anyOf(Stream.of(nodes)
            .map(node -> node.messageSync.get())
            .toArray(CompletableFuture[]::new))
            .whenComplete((o, throwable) -> {
                sync.countDown();
            });
        return sync;
    }

    private void registerNodes(TestRemoteNode... nodes) throws InterruptedException {
        cluster.register(nodes);
        awaitAct();
    }

    private void awaitAllShardAssigned(TestRemoteNode... nodes) throws InterruptedException {
        Set<String> expected = shardsHolder.getShards();
        do {
            Set<String> assigned = new HashSet<>();
            for (int index =0 ; index < nodes.length; index++) {
                assigned.addAll(Arrays.asList(nodes[index].getAssignedShards()));
            }

            var absent = Sets.difference(expected, assigned)
                .stream()
                .sorted()
                .toArray(String[]::new);

            if (absent.length == 0) {
                break;
            }

            logger.info("wait assign next shards: {}", Arrays.toString(absent));
            awaitAct();
        } while (true);

        for (TestRemoteNode node : nodes) {
            logger.info("{} assignments: {}", node.getName(), node.getAssignedShards());
        }
    }

    private void awaitAllShardUnassigned(TestRemoteNode... nodes) throws InterruptedException {
        do {
            int assigned = 0;
            CountDownLatch sync = onMessageSync(nodes);
            for (int index =0 ; index < nodes.length; index++) {
                assigned += nodes[index].assignments.size();
            }

            if (assigned == 0) {
                break;
            }

            sync.await();
        } while (true);
    }

    private void assignAllShards(TestRemoteNode... nodes) {
        for (var shardId : allShards()) {
            TestRemoteNode node = nodes[ThreadLocalRandom.current().nextInt(nodes.length)];
            node.assignments.put(shardId, new AssignmentSeqNo(0, 0));
        }
    }

    private void awaitUpdateState(TestRemoteNode... nodes) throws InterruptedException {
        awaitUpdateState(1, nodes);
    }

    private void awaitUpdateState(int count, TestRemoteNode... nodes) throws InterruptedException {
        for (int index = 0; index < count; index++) {
            for (var node : nodes) {
                node.updateRemoteState();
            }
            awaitAct();
        }
    }

    private BalancerImpl createLeader(int seqNo) {
        return new BalancerImpl(
            clock,
            seqNo,
            List.of(CPU, MEMORY, SHARDS_COUNT),
            cluster,
            executorService,
            ForkJoinPool.commonPool(),
            shardsHolder,
            shardCounter,
            dao);
    }

    private class TestRemoteNode implements RemoteNode {
        public final String name;
        public final AtomicReference<RemoteNodeState> state = new AtomicReference<>();
        public volatile double failedCommands;
        public volatile boolean closed;

        long utime = 100;
        long memory = 1024;
        long startedAt = clock.millis();
        ConcurrentMap<String, AssignmentSeqNo> assignments = new ConcurrentHashMap<>();
        volatile Status assignStatus = Status.OK;
        final AtomicReference<CompletableFuture<?>> messageSync = new AtomicReference<>(new CompletableFuture<>());

        public TestRemoteNode(String name) {
            this.name = name;
        }

        public void updateRemoteState() {
            ThreadLocalRandom random = ThreadLocalRandom.current();
            var memBaseline = MEMORY.maximum(List.of(), List.of()) / 64;
            var cpuBaseline = CPU.maximum(List.of(), List.of()) / 64;

            var result = new RemoteNodeState();
            result.address = name;
            result.memoryBytes = memory;
            result.utimeMillis = utime;
            result.uptimeMillis = clock.millis() - startedAt;
            result.shards = new ArrayList<>(assignments.size());
            for (var shardId : assignments.keySet()) {
                var shard = new RemoteShardState();
                shard.shardId = shardId;
                shard.status = ShardStatus.READY;
                shard.resources = new Resources();
                shard.resources.set(MEMORY, memBaseline * random.nextDouble(1, 1.25));
                shard.resources.set(CPU, cpuBaseline * random.nextDouble(1, 1.25));
                result.shards.add(shard);
            }
            result.receivedAt = clock.millis();
            this.state.set(result);
        }

        @Override
        public String getAddress() {
            return name;
        }

        public String getName() {
            return name;
        }

        @Nullable
        @Override
        public RemoteNodeState takeState() {
            return state.getAndSet(null);
        }

        @Override
        public CompletableFuture<Void> assignShard(String shardId, AssignmentSeqNo seqNo, long expiredAt) {
            return CompletableFuture.runAsync(() -> {
                if (!assignStatus.isOk()) {
                    throw assignStatus.asRuntimeException();
                }

                assignments.put(shardId, seqNo);
            }).whenComplete((ignore, e) -> completeMessage());
        }

        @Override
        public CompletableFuture<Void> unassignShard(String shardId, AssignmentSeqNo seqNo, boolean graceful, long expiredAt) {
            return CompletableFuture.runAsync(() -> {
                var prev = assignments.remove(shardId);
                if (prev == null) {
                    throw Status.NOT_FOUND
                            .withDescription("Unknown shard: " + shardId)
                            .asRuntimeException();
                }
            });
        }

        @Override
        public double getFailCommandPercent() {
            return failedCommands;
        }

        @Override
        public void close() {
            closed = true;
        }

        private void completeMessage() {
            messageSync.getAndSet(new CompletableFuture<>()).completeAsync(() -> {
                return null;
            });
        }

        String[] getAssignedShards() {
            return assignments.keySet()
                .stream()
                .sorted()
                .toArray(String[]::new);
        }

        public void reset() {
            assignments.clear();
            startedAt = clock.millis();
        }
    }
}
