package ru.yandex.solomon.balancer;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.balancer.remote.RemoteNodeState;
import ru.yandex.solomon.balancer.remote.RemoteNodeStub;
import ru.yandex.solomon.ut.ManualClock;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.balancer.ClusterState.countNearExpire;
import static ru.yandex.solomon.balancer.ClusterState.countUnknown;
import static ru.yandex.solomon.balancer.ClusterState.uptimeMillis;

/**
 * @author Vladimir Gordiychuk
 */
public class ClusterStateTest {

    private ManualClock clock;

    @Before
    public void setUp() {
        clock = new ManualClock();
    }

    @Test
    public void clusterUptimeForEmptyCluster() {
        assertClusterUptime(0, List.of());
    }

    @Test
    public void clusterUptimeOneNode() {
        long expected = 42_000;
        NodeSummary node = node("alice");
        setUptime(node, expected);

        assertClusterUptime(expected, List.of(node));
    }

    @Test
    public void clusterUptimeMedian() {
        NodeSummary alice = node("alice");
        NodeSummary bob = node("bob");
        NodeSummary eva = node("eva");
        NodeSummary mark = node("mark");

        {
            setUptime(alice, 2_000);
            setUptime(bob, 5_000);

            assertClusterUptime(2_000, List.of(alice, bob));
        }

        {
            setUptime(alice, 2_000);
            setUptime(bob, 5_000);
            setUptime(eva, 35_000);
            assertClusterUptime(5_000, List.of(alice, bob, eva));
        }

        {
            setUptime(alice, 1_000);
            setUptime(bob, 2_000);
            setUptime(eva, 3_000);
            setUptime(mark, 4_000);
            assertClusterUptime(2_000, List.of(alice, bob, eva, mark));
        }
    }

    @Test
    public void clusterUptimeIgnoreExpiredAnUnknown() {
        var alice = node("alice");
        var bob = node("bob");
        var eva = node("eva");

        alice.markExpired();
        setUptime(eva, 42_000);

        assertEquals(NodeStatus.UNKNOWN, bob.getStatus());
        assertEquals(NodeStatus.EXPIRED, alice.getStatus());
        assertEquals(NodeStatus.CONNECTED, eva.getStatus());

        assertClusterUptime(42_000, List.of(alice, bob, eva));
    }

    @Test
    public void countUnknownNodes() {
        assertEquals(0, countUnknown(List.of()));
        assertEquals(1, countUnknown(List.of(node("alice"))));
        assertEquals(2, countUnknown(List.of(node("1"), node("2"))));
    }

    @Test
    public void countUnknownIgnoreInactive() {
        var alice = node("alice");
        var bob = node("bob");
        var eva = node("eva");

        alice.markExpired();
        setUptime(eva, 42_000);

        assertEquals(NodeStatus.UNKNOWN, bob.getStatus());
        assertEquals(NodeStatus.EXPIRED, alice.getStatus());
        assertEquals(NodeStatus.CONNECTED, eva.getStatus());

        assertEquals(1, countUnknown(List.of(alice, bob, eva)));
    }

    @Test
    public void countNearExpireEmpty() {
        assertEquals(0, countNearExpire(List.of(), 10_000, clock.millis()));
    }

    @Test
    public void countNearExpireAlreadyExpired() {
        var alice = node("alice");
        alice.markExpired();
        assertEquals(1, countNearExpire(List.of(alice), 10_000, clock.millis()));
    }

    @Test
    public void countNearExpireConnectedButNearExpire() {
        var alice = node("alice");
        setUptime(alice, 10_000);
        alice.setExpiredAt(clock.millis() + 1_000);
        assertEquals(1, countNearExpire(List.of(alice), 2_000, clock.millis()));
    }

    @Test
    public void countNearExpireConnectedConnected() {
        var alice = node("alice");
        setUptime(alice, 10_000);
        alice.setExpiredAt(clock.millis() + 30_000);
        assertEquals(0, countNearExpire(List.of(alice), 10_000, clock.millis()));
    }

    private NodeSummary node(String address) {
        return new NodeSummary(new RemoteNodeStub(address));
    }

    private void assertClusterUptime(long expected, List<NodeSummary> nodes) {
        var copy = new ArrayList<>(nodes);
        Collections.shuffle(copy);
        assertEquals(expected, uptimeMillis(copy));
    }

    private void setUptime(NodeSummary node, long uptimeMillis) {
        var state = new RemoteNodeState();
        state.uptimeMillis = uptimeMillis;
        state.receivedAt = clock.millis();
        node.updateNodeState(state);
    }
}
