package ru.yandex.solomon.balancer;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Test;

import ru.yandex.solomon.balancer.remote.RemoteNodeState;
import ru.yandex.solomon.balancer.remote.RemoteNodeStub;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class CommonResourceTest {

    @Test
    public void maximumZeroShards() {
        assertEquals(0, max(List.of(), List.of()), 0);
        assertEquals(0, max(List.of(), List.of(activeNode("alice"))), 0);
        assertEquals(0, max(List.of(), List.of(activeNode("alice"), activeNode("bob"), activeNode("eva"))), 0);
    }

    @Test
    public void oneShard() {
        var shards = List.of(shard("test"));
        var nodes = new ArrayList<NodeSummary>();
        assertEquals(1, max(shards, nodes), 0);
        for (int index = 0; index < 5; index++) {
            nodes.add(activeNode("" + index));
            assertEquals("Nodes " + nodes.size(), 1, max(shards, nodes), 0);
        }
    }

    @Test
    public void twoShard() {
        var shards = List.of(shard("test-1"), shard("test-2"));
        var nodes = new ArrayList<NodeSummary>();
        assertEquals(2, max(shards, nodes), 0);
        for (int index = 0; index < 3; index++) {
            nodes.add(activeNode("" + index));
            assertEquals("Nodes " + nodes.size(), 2, max(shards, nodes), 0);
        }
        nodes.add(activeNode("" + 4));
        assertEquals("Nodes 4", 1, max(shards, nodes), 0);
    }

    @Test
    public void ignoreInactive() {
        var shards = IntStream.range(0, 10)
                .mapToObj(i -> shard("" + i))
                .collect(Collectors.toList());
        var nodes = List.of(activeNode("alice"), inactiveNode("bob"));
        assertEquals(10, max(shards, nodes), 0);
    }

    private static double max(List<ShardSummary> shards, List<NodeSummary> nodes) {
        return CommonResource.SHARDS_COUNT.maximum(shards, nodes);
    }

    private NodeSummary activeNode(String address) {
        var node = new NodeSummary(new RemoteNodeStub(address));
        node.updateNodeState(new RemoteNodeState());
        node.setActive(true);
        return node;
    }

    private NodeSummary inactiveNode(String address) {
        var node = activeNode(address);
        node.setActive(false);
        return node;
    }

    private ShardSummary shard(String id) {
        return new ShardSummary(id);
    }
}
