package ru.yandex.solomon.balancer.dao;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.RandomStringUtils;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.balancer.BalancerOptions;
import ru.yandex.solomon.balancer.CommonResource;
import ru.yandex.solomon.balancer.Resources;
import ru.yandex.solomon.balancer.snapshot.SnapshotAssignments;
import ru.yandex.solomon.balancer.snapshot.SnapshotNode;
import ru.yandex.solomon.balancer.snapshot.SnapshotShard;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class BalancerDaoTest {
    @Before
    public void setUp() {
        getDao().createSchema().join();
    }

    @Test
    public void defaultAssignmentState() {
        var state = getDao().getAssignments().join();
        assertNotNull(state);
    }

    @Test
    public void defaultOpts() {
        var opts = getDao().getOptions().join();
        assertNotNull(opts);
    }

    @Test
    public void saveReadOpts() {
        Resources limits = new Resources();
        limits.set(CommonResource.CPU, 50);
        limits.set(CommonResource.MEMORY, 10 << 10);
        limits.set(CommonResource.SHARDS_COUNT, 100);
        var expected = BalancerOptions.newBuilder()
                .setVersion(100500)
                .setGracefulUnassignExpiration(5, TimeUnit.SECONDS)
                .setForceUnassignExpiration(1, TimeUnit.SECONDS)
                .setLimits(limits)
                .build();

        getDao().saveOptions(expected).join();
        var result = getDao().getOptions().join();
        assertEquals(expected, result);
    }

    @Test
    public void saveReadAssignmentState() {
        var node = new SnapshotNode();
        node.address = "solomon-test";
        node.freeze = true;
        node.active = true;

        var shard = new SnapshotShard();
        shard.shardId = "42";
        shard.node = "solomon-test";
        shard.resources = new Resources();
        shard.resources.set(CommonResource.CPU, 0.2);
        shard.resources.set(CommonResource.MEMORY, 1 << 10);
        shard.resources.set(CommonResource.RECORDS_WRITE_RATE, 5.1);
        shard.resources.set(CommonResource.METRICS_READ_RATE, 10.5);

        var snapshot = new SnapshotAssignments(List.of(node), List.of(shard));

        getDao().saveAssignments(snapshot).join();
        var result = getDao().getAssignments().join();
        assertAssignmentsEqual(snapshot, result);
    }

    @Test
    public void replaceOpts() {
        Resources limits = new Resources();
        limits.set(CommonResource.CPU, 50);
        limits.set(CommonResource.MEMORY, 10 << 10);
        limits.set(CommonResource.SHARDS_COUNT, 100);
        var v1 = BalancerOptions.newBuilder()
                .setVersion(100500)
                .setGracefulUnassignExpiration(5, TimeUnit.SECONDS)
                .setForceUnassignExpiration(1, TimeUnit.SECONDS)
                .setLimits(limits)
                .build();

        getDao().saveOptions(v1).join();

        var v2 = v1.toBuilder()
                .setVersion(100501)
                .build();

        getDao().saveOptions(v2).join();
        var result = getDao().getOptions().join();
        assertEquals(v2, result);
    }

    @Test
    public void replaceAssignments() {
        SnapshotAssignments v1;
        {
            var node = new SnapshotNode();
            node.address = "bob";
            node.freeze = false;
            node.active = true;

            var shard = new SnapshotShard();
            shard.shardId = "42";
            shard.node = "bob";
            shard.resources = new Resources();
            shard.resources.set(CommonResource.CPU, 0.5);
            shard.resources.set(CommonResource.MEMORY, 1 << 10);
            shard.resources.set(CommonResource.RECORDS_WRITE_RATE, 5.1);
            shard.resources.set(CommonResource.METRICS_READ_RATE, 10.5);

            v1 = new SnapshotAssignments(List.of(node), List.of(shard));
        }

        getDao().saveAssignments(v1).join();

        SnapshotAssignments v2;
        {
            var shard = new SnapshotShard();
            shard.shardId = "43";
            shard.node = "bob";
            shard.resources = new Resources();
            shard.resources.set(CommonResource.CPU, 0.6);
            shard.resources.set(CommonResource.MEMORY, 2 << 10);
            shard.resources.set(CommonResource.RECORDS_WRITE_RATE, 10.1);
            shard.resources.set(CommonResource.METRICS_READ_RATE, 20.2);

            v2 = new SnapshotAssignments(v1.nodes, List.of(v1.shards.get(0), shard));
        }

        getDao().saveAssignments(v2).join();
        var result = getDao().getAssignments().join();
        assertAssignmentsEqual(v2, result);
    }

    @Test
    public void hugeAssignmentState() {
        var v1 = randomAssignment(128);
        getDao().saveAssignments(v1).join();
        assertAssignmentsEqual(v1, getDao().getAssignments().join());

        var v2 = randomAssignment(30 << 20); // 30 Mib
        getDao().saveAssignments(v2).join();
        assertAssignmentsEqual(v2, getDao().getAssignments().join());

        var v4 = randomAssignment( 128);
        getDao().saveAssignments(v4).join();
        assertAssignmentsEqual(v4, getDao().getAssignments().join());
    }

    private void assertAssignmentsEqual(SnapshotAssignments expected, SnapshotAssignments actual) {
        assertArrayEquals(expected.nodes.toArray(), actual.nodes.toArray());
        assertArrayEquals(expected.shards.toArray(), actual.shards.toArray());
    }

    private SnapshotAssignments randomAssignment(int expectedBytes) {
        var random = ThreadLocalRandom.current();

        var hostPrefix = RandomStringUtils.randomAlphabetic(200);
        var nodes = new ArrayList<SnapshotNode>();
        for (int index = 0; index < 1_000; index++) {
            var node = new SnapshotNode();
            node.address = hostPrefix + "-" + index + ".yandex-team.ru";
            node.active = random.nextBoolean();
            node.freeze = random.nextBoolean();
            nodes.add(node);
        }

        int totalBytes = 1_000 * 200;
        int shardSize = 256;
        var shardPrefix = RandomStringUtils.randomAlphabetic(256);
        var shards = new ArrayList<SnapshotShard>();
        while (totalBytes < expectedBytes) {
            var shard = new SnapshotShard();
            shard.node = nodes.get(0).address;
            shard.assignmentSeqNo = new AssignmentSeqNo(123, 5555L);
            shard.shardId = shardPrefix + totalBytes;
            shard.resources = new Resources();
            shard.resources.set(CommonResource.ALERTS_COUNT, 42);
            shard.resources.set(CommonResource.SHARDS_COUNT, 1);
            shards.add(shard);
            totalBytes += shardSize;
        }
        return new SnapshotAssignments(nodes, shards);
    }

    protected abstract BalancerDao getDao();
}
