package ru.yandex.solomon.balancer.dao;

import java.util.Arrays;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;

import com.google.protobuf.ByteString;
import com.google.protobuf.UnsafeByteOperations;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;

import ru.yandex.solomon.kikimr.LocalKikimr;
import ru.yandex.solomon.kikimr.YdbHelper;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbBlobDaoTest {

    @ClassRule
    public static LocalKikimr kikimr = new LocalKikimr();

    @Rule
    public TestName testName = new TestName();
    private YdbBlobDao dao;

    @Before
    public void setUp() {
        var ydb = new YdbHelper(kikimr, this.getClass().getSimpleName() + "_" + testName.getMethodName());
        dao = new YdbBlobDao(ydb.getRootPath(), "Test", ydb.getTableClient(), ydb.getSchemeClient());
        dao.createSchema().join();
    }

    @Test
    public void notExistKey() {
        var result = dao.get("not_exist_key").join();
        assertEquals(Optional.empty(), result);
    }

    @Test
    public void putSmall() {
        var value = ByteString.copyFromUtf8("hi");
        dao.put("test", value).join();
        var result = dao.get("test").join();
        assertTrue(result.isPresent());
        assertEquals(value, result.get());
    }

    @Test
    public void putEmpty() {
        var value = ByteString.EMPTY;
        dao.put("test", value).join();
        var result = dao.get("test").join();
        assertTrue(result.isPresent());
        assertEquals(value, result.get());
    }

    @Test
    public void differentKey() {
        var one = ByteString.copyFromUtf8("1");
        var two = ByteString.copyFromUtf8("2");
        dao.put("one", one).join();
        dao.put("two", two).join();

        assertEquals(Optional.of(one), dao.get("one").join());
        assertEquals(Optional.of(two), dao.get("two").join());
    }

    @Test
    public void putHugeValue() {
        var value = randomValue(100 << 20); // 100 MiB
        dao.put("huge", value).join();
        var result = dao.get("huge").join();
        assertTrue(result.isPresent());
        assertEquals(value, result.get());
    }

    @Test
    public void replace() {
        var values = Arrays.asList(
                ByteString.copyFromUtf8("one"),
                ByteString.copyFromUtf8("two"),
                randomValue(80 << 20), // 80 MiB
                randomValue(125 << 20), // 125 MiB
                randomValue(100),
                ByteString.EMPTY
        );

        for (var value : values) {
            dao.put("key", value).join();
            var result = dao.get("key").join();
            assertTrue(result.isPresent());
            assertEquals(value, result.get());
        }
    }

    @Test
    public void deleteNotExist() {
        dao.remove("not_exist").join();
        assertEquals(Optional.empty(), dao.get("not_exist").join());
    }

    @Test
    public void deleteEmptyValue() {
        var value = ByteString.EMPTY;

        dao.put("test", value).join();
        assertEquals(Optional.of(value), dao.get("test").join());

        dao.remove("test").join();
        assertEquals(Optional.empty(), dao.get("test").join());
    }

    @Test
    public void deleteHuge() {
        var value = randomValue(80 << 20); // 80 Mib

        dao.put("test", value).join();
        dao.remove("test").join();
        assertEquals(Optional.empty(), dao.get("test").join());
    }

    private ByteString randomValue(int size) {
        byte[] content = new byte[size];
        ThreadLocalRandom.current().nextBytes(content);
        return UnsafeByteOperations.unsafeWrap(content);
    }
}
