package ru.yandex.solomon.cloud.resource.resolver;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.cloud.resourcemanager.ResolvedFolder;
import ru.yandex.cloud.resourcemanager.ResourceManagerClient;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.NotFoundException;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.misc.concurrent.CompletableFutures.unwrapCompletionException;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class CloudByFolderResolverImpl implements CloudByFolderResolver {

    private static final Logger logger = LoggerFactory.getLogger(CloudByFolderResolverImpl.class);

    private static final long CACHE_POSITIVE_TTL_MINUTES = TimeUnit.HOURS.toMinutes(2L);
    private static final long CACHE_NEGATIVE_TTL_MINUTES = 1;
    private static final long CACHE_MAX_SIZE = 1_000_000;

    private final ResourceManagerClient resourceManagerClient;
    private final Cache<String, String> cloudIdByFolderIdCache;
    private final Cache<String, Throwable> negativeFolderIdsCache;
    private final CloudByFolderResolverMetrics metrics;

    @Autowired
    public CloudByFolderResolverImpl(ResourceManagerClient resourceManagerClient) {
        this.cloudIdByFolderIdCache = CacheBuilder.newBuilder()
            .expireAfterAccess(CACHE_POSITIVE_TTL_MINUTES, TimeUnit.MINUTES)
            .maximumSize(CACHE_MAX_SIZE)
            .build();
        this.negativeFolderIdsCache = CacheBuilder.newBuilder()
            .expireAfterWrite(CACHE_NEGATIVE_TTL_MINUTES, TimeUnit.MINUTES)
            .maximumSize(CACHE_MAX_SIZE)
            .build();
        this.metrics = new CloudByFolderResolverMetrics();
        this.resourceManagerClient = resourceManagerClient;
    }

    @Override
    public CompletableFuture<String> resolveCloudId(String folderId) {
        String cloudId = cloudIdByFolderIdCache.getIfPresent(folderId);
        if (cloudId != null) {
            metrics.hitOk();
            return completedFuture(cloudId);
        }

        Throwable throwable = negativeFolderIdsCache.getIfPresent(folderId);
        if (throwable != null) {
            if (throwable instanceof NotFoundException) {
                metrics.hitNotFound();
                return failedFuture(throwable);
            }
            metrics.hitError();
            return previousFailNotYetExpired();
        }

        metrics.miss();

        return resourceManagerClient.resolve(folderId)
                .thenApply(folderO -> {
                    if (folderO.isPresent()) {
                        return folderO.get().getCloudId();
                    }
                    throw notFoundException(folderId);
                })
            .whenComplete((resolvedCloudId, t) -> {
                if (t != null) {
                    var cause = unwrapCompletionException(t);
                    logger.error("cannot resolve cloud id by folder id {}", folderId, t);
                    negativeFolderIdsCache.put(folderId, cause);
                } else {
                    cloudIdByFolderIdCache.put(folderId, resolvedCloudId);
                }
            });
    }

    @Override
    public CompletableFuture<Optional<String>> tryResolveCloudId(String folderId) {
        var future = new CompletableFuture<Optional<String>>();

        resolveCloudId(folderId)
            .whenComplete((cloudId, t) -> {
                if (t != null) {
                    if (unwrapCompletionException(t) instanceof NotFoundException) {
                        future.complete(Optional.empty());
                    } else {
                        future.completeExceptionally(t);
                    }
                } else {
                    future.complete(Optional.of(cloudId));
                }
            });

        return future;
    }

    @Override
    public CompletableFuture<List<ResolvedFolder>> resolveClouds(Collection<String> folderIds) {
        List<ResolvedFolder> result = new ArrayList<>(folderIds.size());
        Set<String> unresolved = new HashSet<>(folderIds.size());
        for (var folderId : folderIds) {
            String cloudId = cloudIdByFolderIdCache.getIfPresent(folderId);
            if (cloudId != null) {
                result.add(new ResolvedFolder(folderId, cloudId));
            } else {
                unresolved.add(folderId);
            }
        }

        metrics.hitOk(result.size());
        metrics.miss(unresolved.size());
        if (unresolved.isEmpty()) {
            return completedFuture(result);
        }

        return resourceManagerClient.resolve(unresolved)
                .thenApply(resolved -> {
                    for (var folder : resolved) {
                        unresolved.remove(folder.getId());
                        cloudIdByFolderIdCache.put(folder.getId(), folder.getCloudId());
                    }

                    result.addAll(resolved);
                    return result;
                });
    }

    private static NotFoundException notFoundException(String folderId) {
        return new NotFoundException("folder not found by id: " + folderId);
    }

    private static <Y> CompletableFuture<Y> previousFailNotYetExpired() {
        String message = "previous cloud id by folder id resolving fail not yet expired";
        return failedFuture(new BadRequestException(message));
    }
}
