package ru.yandex.solomon.cloud.resource.resolver;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.cloud.resourcemanager.Cloud;
import ru.yandex.cloud.resourcemanager.ResourceManagerClient;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;

/**
 * Special component to fetch cloud projects and rename cloud project names
 *
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class CloudProjectNamesUpdater {

    private static final Duration START_DELAY = Duration.ofMinutes(5);
    private static final Duration ITERATION_DELAY = Duration.ofHours(1);

    private final Logger logger = LoggerFactory.getLogger(CloudProjectNamesUpdater.class);

    private final ScheduledExecutorService executorService;
    private final ResourceManagerClient resourceManagerClient;
    private final ProjectsDao projectsDao;
    private final FeatureFlagsHolder featureFlagsHolder;
    private final String flagProject;

    private volatile Map<String, Cloud> prevClouds = Map.of();

    @Autowired
    public CloudProjectNamesUpdater(
            ThreadPoolProvider tpProvider,
            ResourceManagerClient resourceManagerClient,
            ProjectsDao projectsDao,
            FeatureFlagsHolder featureFlagsHolder,
            @Qualifier("ClientId") Optional<String> clientId)
    {
        this.executorService = tpProvider.getSchedulerExecutorService();
        this.resourceManagerClient = resourceManagerClient;
        this.projectsDao = projectsDao;
        this.featureFlagsHolder = featureFlagsHolder;
        this.flagProject = clientId.orElse("");

        executorService.schedule(this::iteration, START_DELAY.toMillis(), TimeUnit.MILLISECONDS);
    }

    public void iteration() {
        renameCloudProjectNames().whenComplete((aVoid, throwable) -> {
            if (throwable != null) {
                logger.warn("cannot rename cloud project names", throwable);
            }

            long jitter = ThreadLocalRandom.current().nextLong(START_DELAY.toMillis());
            executorService.schedule(this::iteration, ITERATION_DELAY.toMillis() + jitter, TimeUnit.MILLISECONDS);
        });
    }

    private CompletableFuture<Void> renameCloudProjectNames() {
        if (featureFlagsHolder.hasFlag(FeatureFlag.DISABLE_PM_CLOUD_FETCHING, flagProject)) {
            logger.info(FeatureFlag.DISABLE_PM_CLOUD_FETCHING + " disabled for " + flagProject);
            return CompletableFuture.completedFuture(null);
        }
        long startTimeMillis = System.currentTimeMillis();

        Map<String, Cloud> newClouds = new HashMap<>();
        return resourceManagerClient.allClouds().thenCompose(clouds -> {
            List<Cloud> toUpsert = new ArrayList<>();

            Map<String, Cloud> prevClouds = this.prevClouds;
            for (Cloud cloud : clouds) {
                Cloud prevCloud = prevClouds.get(cloud.id());
                if (prevCloud == null) {
                    toUpsert.add(cloud);
                } else if (!prevCloud.isEquals(cloud)) {
                    toUpsert.add(cloud);
                }
                newClouds.put(cloud.id(), cloud);
            }

            if (toUpsert.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }

            List<Project> cloudProjects = toUpsert.stream()
                .map(cloud -> Project.newBuilder()
                    .setId(cloud.id())
                    .setName(cloud.name())
                    .setOnlyAuthPush(true)
                    .setOwner("robot-solomon")
                    .build())
                .collect(Collectors.toList());

            return projectsDao.upsertProjects(cloudProjects);
        })
        .handle((aVoid, throwable) -> {
            if (throwable != null) {
                logger.warn("cannot fetch clouds and rename cloud projects", throwable);
            } else {
                this.prevClouds = ImmutableMap.copyOf(newClouds);
                logger.info("fetch cloud folders and renaming done within {}ms", System.currentTimeMillis() - startTimeMillis);
            }
            return null;
        });
    }
}
