package ru.yandex.solomon.cloud.resource.resolver;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;

import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.labels.shard.ShardKey;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public final class FolderResolverImpl implements FolderResolver {

    private final SolomonConfHolder confHolder;
    private final CloudByFolderResolver cloudByFolderResolver;

    public FolderResolverImpl(
        SolomonConfHolder confHolder,
        CloudByFolderResolver cloudByFolderResolver)
    {
        this.confHolder = confHolder;
        this.cloudByFolderResolver = cloudByFolderResolver;
    }

    @Override
    public CompletableFuture<String> resolveFolderId(ShardKey shardKey) {
        var conf = confHolder.getConfOrThrow();
        var shardConf = conf.findShardOrNull(shardKey);
        if (shardConf != null) {
            var folderId = Strings.nullToEmpty(shardConf.getCluster().getRaw().getFolderId());
            return completedFuture(folderId);
        }

        return resolveFolderId(shardKey.getProject(), shardKey.getCluster());
    }

    @Override
    public CompletableFuture<String> resolveFolderId(String projectId, String clusterName) {
        final var maybeCloudId = projectId;
        final var maybeFolderId = clusterName;

        return cloudByFolderResolver.tryResolveCloudId(maybeFolderId)
            .thenApply(cloudId -> {
                if (cloudId.isEmpty()) {
                    return "";
                }

                var resolvedCloudId = cloudId.get();
                if (!resolvedCloudId.equals(maybeCloudId)) {
                    throw new BadRequestException(
                        String.format(
                            "given folder belongs to other cloud: " +
                                "givenFolderId=%s, givenCloudId=%s, otherCloudId=%s",
                            maybeFolderId,
                            maybeCloudId,
                            resolvedCloudId));
                }

                return maybeFolderId;
            });
    }
}
