package ru.yandex.cluster.discovery;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.net.HostAndPort;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

/**
 * @author Vladimir Gordiychuk
 */
public class ClusterDiscoveryImpl<T extends AutoCloseable> implements ClusterDiscovery<T>, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ClusterDiscoveryImpl.class);

    private final Function<HostAndPort, T> makeTransport;
    private final List<String> unresolvedAddresses;
    private final DiscoveryService discoveryService;
    private final Executor executor;

    private final PingActorRunner actor;
    private final ArrayListLockQueue<Runnable> onChangeCallbacks = new ArrayListLockQueue<>();
    private volatile Map<String, T> transportByFqdn = Collections.emptyMap();
    private volatile List<String> transportList = List.of();
    private volatile boolean closed;

    public ClusterDiscoveryImpl(
        Function<HostAndPort, T> makeTransport,
        List<String> unresolvedAddresses,
        DiscoveryService discoveryService,
        ScheduledExecutorService timer,
        Executor executor,
        long reloadIntervalMillis)
    {
        this.makeTransport = makeTransport;
        this.unresolvedAddresses = unresolvedAddresses;
        this.discoveryService = discoveryService;
        this.executor = executor;
        this.actor = PingActorRunner.newBuilder()
                .operation("cluster_discovery")
                .pingInterval(Duration.ofMillis(reloadIntervalMillis))
                .backoffMaxDelay(Duration.ofMinutes(5))
                .timer(timer)
                .executor(executor)
                .onPing(this::act)
                .build();
        actor.forcePing();
    }

    @Override
    public boolean hasNode(String node) {
        boolean result = transportByFqdn.containsKey(node);
        if (!result) {
            // force trigger reload, because it's can be new node
            actor.forcePing();
        }
        return result;
    }

    @Override
    public Set<String> getNodes() {
        return transportByFqdn.keySet();
    }

    @Override
    public T getTransport() {
        var copy = transportList;
        if (copy.isEmpty()) {
            throw Status.NOT_FOUND
                    .withDescription("Absent nodes in cluster")
                    .asRuntimeException();
        }

        var random = ThreadLocalRandom.current();
        return getTransportByNode(transportList.get(random.nextInt(transportList.size())));
    }

    @Override
    @Nullable
    public T getTransportByNode(String node) {
        var copy = transportByFqdn;
        var transport = copy.get(node);
        if (transport == null) {
            // force trigger reload, because it's can be new node
            actor.forcePing();
            throw Status.NOT_FOUND
                    .withDescription("Node " + node + " is absent in cluster: " + unresolvedAddresses)
                    .asRuntimeException();
        }
        return transport;
    }

    @Override
    public CompletableFuture<Void> forceUpdate() {
        return actor.forcePing();
    }

    @Override
    public void callbackOnChange(Runnable oneShotCallback) {
        logger.debug("Subscribe on change: {}", oneShotCallback);
        onChangeCallbacks.enqueue(oneShotCallback);
    }

    private CompletableFuture<Void> act(int attempt) {
        if (closed) {
            transportByFqdn.values().forEach(this::quiteClose);
            actor.close();
            return CompletableFuture.completedFuture(null);
        }

        logger.debug("Discover list of nodes in cluster {}", unresolvedAddresses);
        return discoveryService.resolve(unresolvedAddresses)
                .thenAccept(hosts -> {
                    var fresh = hosts.stream()
                            .collect(Collectors.toMap(HostAndPort::getHost, Function.identity()));

                    if (transportByFqdn.keySet().equals(fresh.keySet())) {
                        return;
                    }

                    for (var entry : transportByFqdn.entrySet()) {
                        if (!fresh.containsKey(entry.getKey())) {
                            logger.info("Node {} excluded from cluster {}", entry.getKey(), unresolvedAddresses);
                            quiteClose(entry.getValue());
                        }
                    }

                    var result = new HashMap<String, T>();
                    for (var entry : fresh.entrySet()) {
                        var transport = transportByFqdn.get(entry.getKey());
                        if (transport == null) {
                            logger.info("Node {} included into cluster {}", entry.getKey(), unresolvedAddresses);
                            transport = makeTransport.apply(entry.getValue());
                        }

                        result.put(entry.getKey(), transport);
                    }

                    transportByFqdn = Map.copyOf(result);
                    transportList = List.copyOf(transportByFqdn.keySet());
                    notifySubscribers();
                });
    }

    private void quiteClose(T transport) {
        try {
            transport.close();
        } catch (Exception e) {
            logger.warn("Unable to close transport {}", transport);
        }
    }

    private void notifySubscribers() {
        executor.execute(() -> {
            ArrayList<Runnable> copy =  onChangeCallbacks.dequeueAll();
            for (Runnable callback : copy) {
                try {
                    callback.run();
                } catch (Throwable e) {
                    logger.error("callback execution failed", e);
                }
            }
        });
    }

    @Override
    public void close() {
        closed = true;
        actor.forcePing();
    }
}
