package ru.yandex.cluster.discovery;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;

import io.grpc.Status;

import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

/**
 * @author Vladimir Gordiychuk
 */
public class ClusterDiscoveryStub<T extends AutoCloseable> implements ClusterDiscovery<T> {
    private final ConcurrentMap<String, T> transportByFqdn = new ConcurrentHashMap<>();
    private final ArrayListLockQueue<Runnable> onChangeCallbacks = new ArrayListLockQueue<>();

    @Override
    public boolean hasNode(String node) {
        return transportByFqdn.containsKey(node);
    }

    @Override
    public Set<String> getNodes() {
        return Set.copyOf(transportByFqdn.keySet());
    }

    @Override
    public T getTransport() {
        var list = List.copyOf(transportByFqdn.values());
        if (list.isEmpty()) {
            return null;
        }
        return list.get(ThreadLocalRandom.current().nextInt(list.size()));
    }

    @Override
    public T getTransportByNode(String node) {
        var transport = transportByFqdn.get(node);
        if (transport == null) {
            throw Status.NOT_FOUND
                    .withDescription("Node " + node + " is absent in cluster")
                    .asRuntimeException();
        }
        return transport;
    }

    @Override
    public CompletableFuture<Void> forceUpdate() {
        return CompletableFuture.completedFuture(null);
    }

    public void add(String node, T client) {
        this.transportByFqdn.put(node, client);
        notifySubscribers();
    }

    public void remove(String node) {
        var transport = this.transportByFqdn.remove(node);
        if (transport != null) {
            notifySubscribers();
        }
    }

    @Override
    public void callbackOnChange(Runnable oneShotCallback) {
        onChangeCallbacks.enqueue(oneShotCallback);
    }

    private void notifySubscribers() {
        ForkJoinPool.commonPool().execute(() -> {
            ArrayList<Runnable> copy =  onChangeCallbacks.dequeueAll();
            for (Runnable callback : copy) {
                callback.run();
            }
        });
    }

    @Override
    public void close() {
    }
}
