package ru.yandex.cluster.discovery;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.net.HostAndPort;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

/**
 * @author Vladimir Gordiychuk
 */
public class ClusterDiscoveryImplTest {

    private ManualClock clock;
    private ManualScheduledExecutorService executorService;
    private DiscoveryServiceStub discoveryServiceStub;
    private ClusterDiscovery clusterDiscovery;

    @Before
    public void setUp() {
        clock = new ManualClock();
        executorService = new ManualScheduledExecutorService(2, clock);

        discoveryServiceStub = new DiscoveryServiceStub();
        clusterDiscovery = new ClusterDiscoveryImpl<Transport>(
                Transport::new,
                ImmutableList.of("myAddress"),
                discoveryServiceStub,
                executorService,
                executorService,
                30_000);
    }

    @After
    public void tearDown() {
        clusterDiscovery.close();
        executorService.shutdownNow();
    }

    @Test
    public void notifyWhenListsChanged() throws InterruptedException {
        String hostOne = "solomon-storage-sas-000.search.yandex.net";
        String hostTwo = "solomon-storage-sas-001.search.yandex.net";

        assertEquals(Set.of(), clusterDiscovery.getNodes());
        // trigger add new hosts
        {
            CountDownLatch sync = new CountDownLatch(1);
            clusterDiscovery.callbackOnChange(sync::countDown);

            discoveryServiceStub.setHosts(hostOne);
            if (!clusterDiscovery.hasNode(hostOne)) {
                awaitTimeReload(sync);
            }

            assertEquals(Set.of(hostOne), clusterDiscovery.getNodes());
        }

        // repeat trigger add new host after some time
        {
            CountDownLatch sync = new CountDownLatch(1);
            clusterDiscovery.callbackOnChange(sync::countDown);

            discoveryServiceStub.setHosts(hostOne, hostTwo);
            if (!clusterDiscovery.hasNode(hostTwo)) {
                awaitTimeReload(sync);
            }

            assertEquals(Set.of(hostOne, hostTwo), clusterDiscovery.getNodes());
        }

        // repeat trigger remove host
        {
            CountDownLatch sync = new CountDownLatch(1);
            clusterDiscovery.callbackOnChange(sync::countDown);

            discoveryServiceStub.setHosts(hostTwo);
            if (clusterDiscovery.getNodes().size() != 1) {
                awaitTimeReload(sync);
            }

            assertEquals(Set.of(hostTwo), clusterDiscovery.getNodes());
        }
    }

    @Test
    public void triggerReload() throws InterruptedException {
        CountDownLatch sync = new CountDownLatch(1);
        clusterDiscovery.callbackOnChange(sync::countDown);

        String host = "solomon-storage-sas-000.search.yandex.net";
        assertEquals(Collections.emptySet(), clusterDiscovery.getNodes());

        discoveryServiceStub.setHosts(host);
        // trigger update
        clusterDiscovery.hasNode(host);
        sync.await();
        assertEquals(ImmutableSet.of(host), clusterDiscovery.getNodes());
    }

    @Test
    public void hasNode() throws InterruptedException {
        CountDownLatch sync = new CountDownLatch(1);
        clusterDiscovery.callbackOnChange(sync::countDown);

        assertFalse(clusterDiscovery.hasNode("notExists"));

        String host = "solomon-storage-sas-000.search.yandex.net";
        discoveryServiceStub.setHosts(host);
        awaitTimeReload(sync);

        assertTrue(clusterDiscovery.hasNode(host));
    }

    @Test
    public void noFound() {
        try {
            clusterDiscovery.getTransportByNode("solomon-storage-sas-000.search.yandex.net");
            fail("expected fail with not found status");
        } catch (StatusRuntimeException e) {
            assertEquals(Status.Code.NOT_FOUND, e.getStatus().getCode());
        }
    }

    @Test
    public void sameAddress() throws InterruptedException {
        String hostOne = "solomon-storage-sas-000.search.yandex.net";
        String hostTwo = "solomon-storage-sas-001.search.yandex.net";

        assertEquals(Set.of(), clusterDiscovery.getNodes());

        CountDownLatch sync = new CountDownLatch(1);
        clusterDiscovery.callbackOnChange(sync::countDown);

        discoveryServiceStub.setHosts(hostOne);
        if (clusterDiscovery.getNodes().isEmpty()) {
            awaitTimeReload(sync);
        }

        var transport = clusterDiscovery.getTransportByNode(hostOne);
        assertNotNull(transport);

        CountDownLatch syncTwo = new CountDownLatch(1);
        clusterDiscovery.callbackOnChange(syncTwo::countDown);

        discoveryServiceStub.setHosts(hostOne, hostTwo);
        if (clusterDiscovery.getNodes().size() == 1) {
            awaitTimeReload(syncTwo);
        }

        assertNotNull(clusterDiscovery.getTransportByNode(hostTwo));
        assertSame(transport, clusterDiscovery.getTransportByNode(hostOne));
    }

    @Test
    public void randomNode() throws InterruptedException {
        String hostOne = "solomon-storage-sas-000.search.yandex.net";
        String hostTwo = "solomon-storage-sas-001.search.yandex.net";

        assertEquals(Set.of(), clusterDiscovery.getNodes());
        {

            CountDownLatch sync = new CountDownLatch(1);
            clusterDiscovery.callbackOnChange(sync::countDown);
            discoveryServiceStub.setHosts(hostOne);
            if (clusterDiscovery.getNodes().isEmpty()) {
                awaitTimeReload(sync);
            }

            var random = clusterDiscovery.getTransport();
            assertSame(random, clusterDiscovery.getTransportByNode(hostOne));
        }

        {
            CountDownLatch syncTwo = new CountDownLatch(1);
            clusterDiscovery.callbackOnChange(syncTwo::countDown);

            discoveryServiceStub.setHosts(hostOne, hostTwo);
            if (clusterDiscovery.getNodes().size() == 1) {
                awaitTimeReload(syncTwo);
            }

            Set<AutoCloseable> uniqueHosts = new HashSet<>();
            for (int index = 0; index < 100; index++) {
                uniqueHosts.add(clusterDiscovery.getTransport());
            }
            assertEquals(2, uniqueHosts.size());
        }
    }

    @Test
    public void localhostAvailableByFqdn() throws InterruptedException {
        CountDownLatch sync = new CountDownLatch(1);
        clusterDiscovery.callbackOnChange(sync::countDown);

        discoveryServiceStub.setHosts("localhost");
        if (clusterDiscovery.getNodes().isEmpty()) {
            awaitTimeReload(sync);
        }

        assertEquals(ImmutableSet.of("localhost"), clusterDiscovery.getNodes());
        assertTrue(clusterDiscovery.hasNode("localhost"));
        assertNotNull(clusterDiscovery.getTransportByNode("localhost"));
    }

    private void awaitTimeReload(CountDownLatch sync) throws InterruptedException {
        do {
            clock.passedTime(1, TimeUnit.MINUTES);
        } while (!sync.await(10, TimeUnit.MICROSECONDS));
    }

    private static class Transport extends DefaultObject implements AutoCloseable {
        private final String address;
        private volatile boolean closed;

        public Transport(HostAndPort hostAndPort) {
            this.address = hostAndPort.getHost();
        }

        @Override
        public void close() {
            closed = true;
        }
    }
}
