package ru.yandex.concurrency.limits.actors;

import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.Executor;

import com.google.common.util.concurrent.MoreExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
public class LimiterActorRunner implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(LimiterActorRunner.class);

    private final Limiter limiter;
    private final String operation;
    private final OperationProvider provider;

    private final ActorRunner actor;
    private volatile boolean closed;

    // for manager ui
    Throwable lastError;
    Instant lastErrorTime = Instant.EPOCH;

    public LimiterActorRunner(Builder builder) {
        this.limiter = Objects.requireNonNull(builder.limiter);
        this.operation = builder.operation;
        this.provider = Objects.requireNonNull(builder.provider);
        this.actor = new ActorRunner(this::act, builder.executor);
        this.limiter.onChange(actor::schedule);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public void schedule() {
        actor.schedule();
    }

    public void addQueueSize(int size) {
        limiter.addQueueSize(size);
    }

    public void addQueueTime(long queueTimeMs) {
        limiter.addQueueTime(queueTimeMs);
    }

    public void addStatus(OperationStatus status) {
        limiter.addStatus(status);
    }

    @Override
    public void close() throws Exception {
        closed = true;
    }

    public int inflight() {
        return limiter.inflight();
    }

    public int limit() {
        return limiter.limit();
    }

    private void act() {
        while (!closed && provider.hasNext()) {
            var permit = limiter.acquire();
            if (permit == null) {
                break;
            }

            runAcquired(permit);
        }
    }

    private void runAcquired(OperationPermit permit) {
        try {
            var future = provider.next();
            // fast pass
            if (future.isDone()) {
                permit.release(future.getNow(OperationStatus.IGNORE));
                return;
            }

            future.whenComplete((status, e) -> {
                if (e != null) {
                    logError(e);
                    status = OperationStatus.DROP;
                }

                permit.release(status);
            });
        } catch (Throwable e) {
            logError(e);
            permit.release(OperationStatus.DROP);
        }
    }

    private void logError(Throwable e) {
        logger.warn("operation {} failed", operation, e);
        lastError = e;
        lastErrorTime = Instant.now();
    }

    public static class Builder {
        private Limiter limiter;
        private String operation = "unknown";
        private Executor executor = MoreExecutors.directExecutor();
        private OperationProvider provider;

        private Builder() {
        }

        public Builder limiter(Limiter limiter) {
            this.limiter = limiter;
            return this;
        }

        public Builder executor(Executor executor) {
            this.executor = executor;
            return this;
        }

        public Builder operation(String operation) {
            this.operation = operation;
            return this;
        }

        public Builder operationProvider(OperationProvider provider) {
            this.provider = provider;
            return this;
        }

        public LimiterActorRunner build() {
            return new LimiterActorRunner(this);
        }
    }
}
