package ru.yandex.concurrency.limits.actors;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.Nullable;

import com.netflix.concurrency.limits.limit.Gradient2Limit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
public class LimiterImpl implements Limiter {
    private static final Logger logger = LoggerFactory.getLogger(LimiterImpl.class);

    private final Gradient2Limit limit;

    private final long minRttNanos;
    private final OperationMetrics metrics;
    private final AtomicInteger inflight = new AtomicInteger();
    private final List<Runnable> listeners = new CopyOnWriteArrayList<>();

    public LimiterImpl(Builder builder) {
        this.limit = builder.limit.build();
        this.minRttNanos = builder.minRttNanos;
        this.metrics = new OperationMetrics(builder.registry);
        limit.notifyOnChange(integer -> notifyOnChange());
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    @Override
    public void addQueueSize(int size) {
        metrics.queueSize.add(size);
    }

    @Override
    public void addQueueTime(long queueTimeMs) {
        metrics.queueTime.record(queueTimeMs);
    }

    @Override
    public void addStatus(OperationStatus status) {
        metrics.statuses.get(status).inc();
    }

    @Override
    public int inflight() {
        return inflight.get();
    }

    @Override
    public int limit() {
        return limit.getLimit();
    }

    @Nullable
    @Override
    public OperationPermit acquire() {
        if (!tryAcquire()) {
            return null;
        }

        return createPermit();
    }

    @Override
    public void onChange(Runnable runnable) {
        listeners.add(runnable);
    }

    private void notifyOnChange() {
        for (var listener : listeners) {
            try {
                listener.run();
            } catch (Throwable e) {
                logger.error("Failed listener {}", listener, e);
            }
        }
    }

    private boolean tryAcquire() {
        int actual;
        do {
            actual = inflight.get();
            if (actual >= limit.getLimit()) {
                return false;
            }
        } while (!inflight.compareAndSet(actual, actual + 1));
        return true;
    }

    private OperationPermit createPermit() {
        int currentInflight = inflight.get();
        long startNanos = System.nanoTime();

        metrics.inflight.record(currentInflight);
        metrics.limit.record(limit.getLimit());
        var done = new AtomicBoolean();

        return status -> {
            if (done.compareAndSet(false, true)) {
                release(startNanos, currentInflight, status);
            }
        };
    }

    private void release(long startNanos, int inflight, OperationStatus status) {
        addStatus(status);
        if (this.inflight.decrementAndGet() < limit.getLimit()) {
            notifyOnChange();
        }

        if (status == OperationStatus.IGNORE) {
            return;
        }

        long rtt = System.nanoTime() - startNanos;
        long rttMax = Math.max(rtt, minRttNanos);
        limit.onSample(startNanos, rttMax, inflight, status == OperationStatus.DROP);
        metrics.rtt.record(TimeUnit.NANOSECONDS.toMillis(rtt));
        metrics.minDifference.record(limit.getRttNoLoad(TimeUnit.NANOSECONDS) / rttMax);
    }

    public static class Builder {
        private final Gradient2Limit.Builder limit = Gradient2Limit.newBuilder();
        private long minRttNanos;
        private MetricRegistry registry = new MetricRegistry();
        private String operation = "unknown";

        private Builder() {
        }

        public Builder minLimit(int min) {
            limit.minLimit(min);
            return this;
        }

        /**
         * All rtt less then specified are good
         */
        public Builder minRtt(long minRttNanos) {
            this.minRttNanos = minRttNanos;
            return this;
        }

        public Builder maxLimit(int max) {
            limit.maxConcurrency(max);
            return this;
        }

        public Builder initLimit(int init) {
            limit.initialLimit(init);
            return this;
        }

        public Builder rttTolerance(double rttTolerance) {
            limit.rttTolerance(rttTolerance);
            return this;
        }

        public Builder operation(String operation) {
            this.operation = operation;
            return this;
        }

        public Builder registry(MetricRegistry registry) {
            this.registry = registry;
            return this;
        }

        public LimiterImpl build() {
            registry = registry.subRegistry("operation", operation);
            limit.metricRegistry(new MonlibMetricRegistry(registry, "operation.limiter"));
            return new LimiterImpl(this);
        }
    }
}
