package ru.yandex.concurrency.limits.actors;


import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import com.netflix.concurrency.limits.MetricIds;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
public class MonlibMetricRegistry implements com.netflix.concurrency.limits.MetricRegistry {
    private final MetricRegistry registry;
    private final String prefix;

    public MonlibMetricRegistry(MetricRegistry registry, String prefix) {
        this.registry = registry;
        this.prefix = prefix;
    }

    @Override
    public SampleListener distribution(String id, String... tagNameValuePairs) {
        var histogram = registry.histogramRate(
                metricName(id),
                toLabels(tagNameValuePairs),
                Histograms.exponential(30, 2));

        switch (id) {
            case MetricIds.MIN_RTT_NAME:
            case MetricIds.WINDOW_MIN_RTT_NAME:
                return new TimeDistributionToHist(histogram);
            default:
                return new DistributionToHist(histogram);
        }
    }

    @Override
    public void gauge(String id, Supplier<Number> supplier, String... tagNameValuePairs) {
        registry.lazyGaugeDouble(metricName(id), toLabels(tagNameValuePairs), () -> supplier.get().doubleValue());
    }

    @Override
    public Counter counter(String id, String... tagNameValuePairs) {
        var rate = registry.rate(metricName(id), toLabels(tagNameValuePairs));
        return new CounterToRate(rate);
    }

    private String metricName(String id) {
        return prefix + "." + id;
    }

    private Labels toLabels(String[] tagNameValuePairs) {
        if (tagNameValuePairs.length == 0) {
            return Labels.empty();
        }

        var builder = Labels.builder(tagNameValuePairs.length / 2);
        for (int index = 0; index < tagNameValuePairs.length; index += 2) {
            builder.add(tagNameValuePairs[index], tagNameValuePairs[index + 1]);
        }
        return builder.build();
    }

    private static record CounterToRate(Rate rate) implements Counter {
        @Override
        public void increment() {
            rate.inc();
        }
    }

    private static record DistributionToHist(Histogram histogram) implements SampleListener {
        @Override
        public void addSample(Number value) {
            histogram.record(value.longValue());
        }
    }

    private static record TimeDistributionToHist(Histogram histogram) implements SampleListener {
        @Override
        public void addSample(Number value) {
            histogram.record(TimeUnit.NANOSECONDS.toMillis(value.longValue()));
        }
    }
}
