package ru.yandex.concurrency.limits.actors;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicIntegerArray;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.Test;

import static org.hamcrest.Matchers.lessThanOrEqualTo;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class LimiterActorRunnerTest {

    private OperationProviderStub operations;

    @Before
    public void setUp() throws Exception {
        operations = new OperationProviderStub();
    }

    @Test
    public void syncOperations() throws InterruptedException {
        var actor = createActor();

        CountDownLatch sync = new CountDownLatch(100);
        for (int index = 0; index < 100; index++) {
            operations.enqueue(() -> {
                sync.countDown();
                return CompletableFuture.completedFuture(OperationStatus.SUCCESS);
            });
        }

        actor.schedule();
        assertTrue(sync.await(3, TimeUnit.SECONDS));
    }

    @Test
    public void asyncOperations() throws InterruptedException {
        var actor = createActor();

        CountDownLatch sync = new CountDownLatch(1000);
        for (int index = 0; index < 1000; index++) {
            operations.enqueue(() -> {
                return CompletableFuture.supplyAsync(() -> {
                    sync.countDown();
                    return OperationStatus.SUCCESS;
                });
            });
        }

        actor.schedule();
        assertTrue(sync.await(3, TimeUnit.SECONDS));
    }

    @Test
    public void limitInflightByMax() throws InterruptedException {
        var actor = LimiterActorRunner.newBuilder()
                .operationProvider(operations)
                .limiter(LimiterImpl.newBuilder()
                        .initLimit(1)
                        .minLimit(1)
                        .maxLimit(2)
                        .build())
                .executor(ForkJoinPool.commonPool())
                .build();

        int count = 1000;
        CountDownLatch sync = new CountDownLatch(count);
        AtomicInteger running = new AtomicInteger();
        AtomicIntegerArray inflight = new AtomicIntegerArray(count);
        for (int index = 0; index < count; index++) {
            int finalIndex = index;
            operations.enqueue(() -> {
                inflight.set(finalIndex, running.incrementAndGet());
                return CompletableFuture.supplyAsync(() -> {
                    try {
                        TimeUnit.NANOSECONDS.sleep(ThreadLocalRandom.current().nextInt(5));
                        return OperationStatus.SUCCESS;
                    } catch (InterruptedException e) {
                        throw new RuntimeException(e);
                    } finally {
                        running.decrementAndGet();
                        sync.countDown();
                    }
                });
            });
        }

        actor.schedule();
        assertTrue(sync.await(3, TimeUnit.SECONDS));

        int maxInflight = IntStream.range(0, count).map(inflight::get).max().getAsInt();
        assertThat(maxInflight, lessThanOrEqualTo(2));
    }

    @Test
    public void failedSyncFutures() throws InterruptedException {
        var actor = createActor();

        CountDownLatch sync = new CountDownLatch(3);
        for (int index = 0; index < 3; index++) {
            operations.enqueue(() -> {
                sync.countDown();
                return CompletableFuture.failedFuture(new RuntimeException("Hi"));
            });
        }

        actor.schedule();
        assertTrue(sync.await(3, TimeUnit.SECONDS));
    }

    @Test
    public void failedAsyncFutures() throws InterruptedException {
        var actor = createActor();

        CountDownLatch sync = new CountDownLatch(3);
        for (int index = 0; index < 3; index++) {
            operations.enqueue(() -> {
                sync.countDown();
                return CompletableFuture.supplyAsync(() -> {
                    throw new RuntimeException("Hi");
                });
            });
        }

        actor.schedule();
        assertTrue(sync.await(3, TimeUnit.SECONDS));
    }

    @Test
    public void failedOperation() throws InterruptedException {
        var actor = createActor();

        CountDownLatch sync = new CountDownLatch(3);
        for (int index = 0; index < 3; index++) {
            operations.enqueue(() -> {
                sync.countDown();
                throw new RuntimeException("Hi");
            });
        }

        actor.schedule();
        assertTrue(sync.await(3, TimeUnit.SECONDS));
    }

    private LimiterActorRunner createActor() {
        return LimiterActorRunner.newBuilder()
                .operationProvider(operations)
                .limiter(LimiterImpl.newBuilder()
                        .initLimit(1)
                        .minLimit(1)
                        .build())
                .executor(ForkJoinPool.commonPool())
                .build();
    }
}
