package ru.yandex.solomon.core.conf;

import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.ClusterServiceAssociation;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;

/**
 * @author Sergey Polovko
 */
@Component
@Import(ConfigDaoContext.class)
public class ClustersManager {

    private final ClustersDao clustersDao;
    private final ProjectsDao projectsDao;
    private final ShardsDao shardsDao;
    private final ShardsManager shardsManager;

    @Autowired
    public ClustersManager(ClustersDao clustersDao, ProjectsDao projectsDao, ShardsDao shardsDao, ShardsManager shardsManager) {
        this.clustersDao = clustersDao;
        this.projectsDao = projectsDao;
        this.shardsDao = shardsDao;
        this.shardsManager = shardsManager;
    }

    public CompletableFuture<PagedResult<Cluster>> getAllClusters(PageOptions pageOpts, String text) {
        return clustersDao.findAll(pageOpts, text);
    }

    public CompletableFuture<PagedResult<Cluster>> getClusters(
        String projectId,
        String folderId,
        PageOptions pageOpts,
        String text)
    {
        return clustersDao.findByProjectId(projectId, folderId, pageOpts, text);
    }

    public CompletableFuture<TokenBasePage<Cluster>> getPagedClusters(
        String projectId,
        String folderId,
        int pageSize,
        String pageToken,
        String text)
    {
        return clustersDao.findByProjectIdPaged(projectId, folderId, pageSize, pageToken, text);
    }

    public CompletableFuture<Cluster> getCluster(String projectId, String folderId, String clusterId) {
        return clustersDao.findOne(projectId, folderId, clusterId)
            .thenApply(cluster -> cluster.orElseThrow(() -> clusterNotFound(projectId, clusterId)));
    }

    public CompletableFuture<Cluster> createCluster(Cluster cluster) {
        return checkForeignRefs(cluster)
            .thenCompose(aVoid -> clustersDao.insert(cluster))
            .thenApply(inserted -> {
                if (inserted) {
                    return cluster;
                }
                throw ConflictException.alreadyExists("cluster", cluster.getProjectId(), cluster.getId());
            });
    }

    public CompletableFuture<Cluster> updateCluster(Cluster cluster) {
        // TODO: use transactions instead of this messy code
        return checkForeignRefs(cluster)
            .thenCompose(aVoid -> clustersDao.findOne(cluster.getProjectId(), "", cluster.getId()))
            .thenCompose(oldClusterOpt -> {
                if (oldClusterOpt.isEmpty()) {
                    throw clusterNotFound(cluster.getProjectId(), cluster.getId());
                }

                Cluster oldCluster = oldClusterOpt.get();
                if (cluster.getVersion() != oldCluster.getVersion()) {
                    throw clusterIsOutOfDate(cluster.getProjectId(), "", cluster.getId(), cluster.getVersion());
                }

                if (Objects.equals(cluster.getName(), oldCluster.getName())) {
                    // if names are equal then just update
                    return doClusterUpdate(cluster);
                }

                CompletableFuture<Cluster> resultFuture = new CompletableFuture<>();
                shardsDao.patchWithClusterName(cluster.getProjectId(), cluster.getId(), cluster.getName())
                    .thenCompose(aVoid2 -> doClusterUpdate(cluster))
                    .whenComplete((updatedCluster, throwable) -> {
                        if (throwable == null) {
                            resultFuture.complete(updatedCluster);
                            return;
                        }
                        // revert shards patch
                        shardsDao.patchWithClusterName(cluster.getProjectId(), cluster.getId(), oldCluster.getName())
                            .whenComplete((aVoid, ignoredException) -> {
                                resultFuture.completeExceptionally(throwable);
                            });
                    });
                return resultFuture;
            });
    }

    private CompletableFuture<Cluster> doClusterUpdate(Cluster cluster) {
        return clustersDao.partialUpdate(cluster)
            .thenApply(updatedCluster -> {
                if (updatedCluster.isPresent()) {
                    return updatedCluster.get();
                }
                throw clusterNotFound(cluster.getProjectId(), cluster.getId());
            });
    }

    public CompletableFuture<Void> deleteCluster(String projectId, String folderId, String clusterId) {
        return checkRelatedServices(projectId, folderId, clusterId)
            .thenCompose(aVoid -> clustersDao.deleteOne(projectId, folderId, clusterId))
            .thenAccept(deleted -> {
                if (!deleted) {
                    throw clusterNotFound(projectId, clusterId);
                }
            });
    }

    private CompletableFuture<Void> checkRelatedServices(String projectId, String folderId, String clusterId) {
        return shardsDao.findByClusterId(projectId, folderId, clusterId)
            .thenAccept(shards -> {
                if (!shards.isEmpty()) {
                    List<String> serviceIds = shards.stream()
                        .map(Shard::getServiceId)
                        .collect(Collectors.toList());
                    String message = "cluster cannot be deleted because it has associated services: " + serviceIds;
                    throw new BadRequestException(message);
                }
            });
    }

    public CompletableFuture<List<ClusterServiceAssociation>> getClusterAssociations(
        String projectId,
        String folderId,
        String clusterId)
    {
        return shardsDao.findByClusterId(projectId, folderId, clusterId)
            .thenApply(shards -> shards.stream()
                .map(ClusterServiceAssociation::createAssociationToService)
                .collect(Collectors.toList()));
    }

    public CompletableFuture<List<ShardsManager.ShardExtended>> getClusterShards(
            String projectId,
            String folderId,
            String clusterId,
            EnumSet<ShardState> state)
    {
        return shardsDao.findByClusterId(projectId, folderId, clusterId)
                .thenApply(shards -> shards.stream()
                        .filter(shard -> state == null || state.isEmpty() || state.contains(shard.getState()))
                        .collect(Collectors.toList()))
                .thenCompose(shards -> shardsManager.getExtendedShards(shards, projectId, folderId, true));
    }

    public CompletableFuture<TokenBasePage<ClusterServiceAssociation>> getClusterAssociations(
            String projectId,
            String folderId,
            String clusterId,
            Set<ShardState> states,
            String filter,
            int pageSize,
            String pageToken)
    {
        return shardsDao.findByClusterIdV3(projectId, folderId, clusterId, states, filter, pageSize, pageToken)
            .thenApply(shards -> shards.map(ClusterServiceAssociation::createAssociationToService));
    }

    private CompletableFuture<Void> checkForeignRefs(Cluster cluster) {
        return projectsDao.exists(cluster.getProjectId())
            .thenAccept(exists -> {
                if (!exists) {
                    throw new BadRequestException(String.format("project %s does not exist", cluster.getProjectId()));
                }
            });
    }

    private static NotFoundException clusterNotFound(String projectId, String clusterId) {
        return new NotFoundException(String.format("no cluster with id '%s' in project '%s'", clusterId, projectId));
    }

    private static ConflictException clusterIsOutOfDate(String projectId, String folderId, String clusterId, int version) {
        String message = String.format(
            "cluster (%s, %s) with version %s is out of date",
            StringUtils.isEmpty(folderId) ? projectId : folderId,
            clusterId,
            version
        );
        return new ConflictException(message);
    }
}
