package ru.yandex.solomon.core.conf;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.DashboardsDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.ShortGraphOrDashboardConf;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

/**
 * @author Oleg Baryshnikov
 */
@Component
@Import(ConfigDaoContext.class)
public class DashboardsManager {

    private final DashboardsDao dashboardsDao;
    private final ProjectsDao projectsDao;

    @Autowired
    public DashboardsManager(DashboardsDao dashboardsDao, ProjectsDao projectsDao) {
        this.dashboardsDao = dashboardsDao;
        this.projectsDao = projectsDao;
    }

    public CompletableFuture<PagedResult<Dashboard>> getDashboards(
        String projectId,
        String folderId,
        PageOptions pageOpts,
        String text)
    {
        return dashboardsDao.findByProjectId(projectId, folderId, pageOpts, text);
    }

    public CompletableFuture<PagedResult<Dashboard>> getAllDashboards(
        PageOptions pageOpts,
        String text)
    {
        return dashboardsDao.findAll(pageOpts, text);
    }

    public CompletableFuture<Dashboard> getDashboard(String projectId, String folderId, String dashboardId) {
        return dashboardsDao.findOne(projectId, folderId, dashboardId)
            .thenApply(dashboard -> dashboard.orElseThrow(() -> dashboardNotFound(projectId, folderId, dashboardId)));
    }

    public CompletableFuture<Dashboard> createDashboard(Dashboard dashboard) {
        return checkForeignRefs(dashboard)
            .thenCompose(aVoid -> dashboardsDao.insert(dashboard))
            .thenApply(inserted -> {
                if (!inserted) {
                    throw ConflictException.alreadyExists("dashboard", dashboard.getProjectId(), dashboard.getId());
                }
                return dashboard;
            });
    }

    public CompletableFuture<Dashboard> updateDashboard(Dashboard dashboard) {
        return checkForeignRefs(dashboard)
            .thenCompose(aVoid -> dashboardsDao.partialUpdate(dashboard))
            .thenCompose(updateDashboard -> {
                //noinspection OptionalIsPresent
                if (updateDashboard.isPresent()) {
                    return CompletableFuture.completedFuture(updateDashboard.get());
                }

                return dashboardsDao.exists(dashboard.getProjectId(), dashboard.getFolderId(), dashboard.getId())
                    .thenApply(exists -> {
                        if (exists) {
                            throw dashboardIsOutOfDate(
                                dashboard.getProjectId(),
                                dashboard.getFolderId(),
                                dashboard.getId(),
                                dashboard.getVersion());
                        }
                        throw dashboardNotFound(dashboard.getProjectId(), dashboard.getFolderId(), dashboard.getId());
                    });
            });
    }

    public CompletableFuture<Void> deleteDashboard(String projectId, String folderId, String dashboardId) {
        return dashboardsDao.deleteOne(projectId, folderId, dashboardId)
            .thenAccept(deleted -> {
                if (!deleted) {
                    throw dashboardNotFound(projectId, folderId, dashboardId);
                }
            });
    }

    public CompletableFuture<List<ShortGraphOrDashboardConf>> getAllDashboardsMatchingLabels(String projectId, String folderId, Labels labels) {
        return dashboardsDao.findByProjectIdShorted(projectId, folderId)
            .thenApply(dashboardItems -> dashboardItems.stream()
                .filter(x -> SelectorUtils.fromProjectAndParams(x.getProjectId(), x.getParameters()).matchesAll(labels))
                .collect(Collectors.toList()));
    }

    private CompletableFuture<Void> checkForeignRefs(Dashboard dashboard) {
        return projectsDao.exists(dashboard.getProjectId())
            .thenAccept(exists -> {
                if (!exists) {
                    throw new BadRequestException(String.format("project %s does not exist", dashboard.getProjectId()));
                }
            });
    }

    private static NotFoundException dashboardNotFound(String projectId, String folderId, String dashboardId) {
        if (!StringUtils.isEmpty(folderId)) {
            return new NotFoundException(String.format("no dashboard with id '%s' in folder '%s'", dashboardId, folderId));
        }
        return new NotFoundException(String.format("no dashboard with id '%s' in project '%s'", dashboardId, projectId));
    }

    private static ConflictException dashboardIsOutOfDate(
        String projectId,
        String folderId,
        String dashboardId,
        int version)
    {
        String message = String.format(
            "dashboard (%s, %s) with version %s is out of date",
            StringUtils.isEmpty(folderId) ? projectId : folderId,
            dashboardId,
            version
        );
        return new ConflictException(message);
    }
}
