package ru.yandex.solomon.core.conf;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.GraphsDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.ShortGraphOrDashboardConf;
import ru.yandex.solomon.core.db.model.graph.Graph;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.labels.selector.LabelSelectorSet;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

/**
 * @author Oleg Baryshnikov
 */
@Component
@Import(ConfigDaoContext.class)
public class GraphsManager {

    private final GraphsDao graphsDao;
    private final ProjectsDao projectsDao;

    @Autowired
    public GraphsManager(GraphsDao graphsDao, ProjectsDao projectsDao) {
        this.graphsDao = graphsDao;
        this.projectsDao = projectsDao;
    }

    public CompletableFuture<PagedResult<Graph>> getGraphs(
        String projectId,
        String folderId,
        PageOptions pageOpts,
        String text)
    {
        return graphsDao.findByProjectId(projectId, folderId, pageOpts, text);
    }

    public CompletableFuture<List<ShortGraphOrDashboardConf>> getAllGraphsMatchingLabels(String projectId, String folderId, Labels labels) {
        return graphsDao.findByProjectIdShorted(projectId, folderId)
            .thenApply(graphs -> graphs.stream()
                .filter(graph -> {
                    LabelSelectorSet labelSelectorSet =
                        SelectorUtils.fromProjectAndParams(graph.getProjectId(), graph.getParameters());
                    return labelSelectorSet.matchesAll(labels);
                })
                .collect(Collectors.toList()));
    }

    public CompletableFuture<PagedResult<Graph>> getAllGraphs(PageOptions pageOpts, String text) {
        return graphsDao.findAll(pageOpts, text);
    }

    public CompletableFuture<Graph> getGraph(String projectId, String folderId, String graphId) {
        return graphsDao.findOne(projectId, folderId, graphId)
            .thenApply(graph -> graph.orElseThrow(() -> graphNotFound(projectId, folderId, graphId)));
    }

    public CompletableFuture<Graph> createGraph(Graph graph) {
        return checkForeignRefs(graph)
            .thenCompose(aVoid -> graphsDao.insert(graph))
            .thenApply(inserted -> {
                if (!inserted) {
                    throw ConflictException.alreadyExists("graph", graph.getProjectId(), graph.getId());
                }
                return graph;
            });
    }

    public CompletableFuture<Graph> updateGraph(Graph graph) {
        return checkForeignRefs(graph)
            .thenCompose(aVoid -> graphsDao.partialUpdate(graph))
            .thenCompose(updatedGraphO -> {
                //noinspection OptionalIsPresent
                if (updatedGraphO.isPresent()) {
                    return CompletableFuture.completedFuture(updatedGraphO.get());
                }

                return graphsDao.exists(graph.getProjectId(), graph.getFolderId(), graph.getId())
                    .thenApply(exists -> {
                        if (exists) {
                            throw graphIsOutOfDate(graph.getProjectId(), graph.getFolderId(), graph.getId(), graph.getVersion());
                        }
                        throw graphNotFound(graph.getProjectId(), graph.getFolderId(), graph.getId());
                    });
            });
    }

    public CompletableFuture<Void> deleteGraph(String projectId, String folderId, String graphId) {
        return graphsDao.deleteOne(projectId, folderId, graphId)
            .thenAccept(deleted -> {
                if (!deleted) {
                    throw graphNotFound(projectId, folderId, graphId);
                }
            });
    }

    private CompletableFuture<Void> checkForeignRefs(Graph graph) {
        return projectsDao.exists(graph.getProjectId())
            .thenAccept(exists -> {
                if (!exists) {
                    throw new BadRequestException(String.format("project %s does not exist", graph.getProjectId()));
                }
            });
    }

    private static NotFoundException graphNotFound(String projectId, String folderId, String graphId) {
        if (!StringUtils.isEmpty(folderId)) {
            return new NotFoundException(String.format("no graph with id '%s' in folder '%s'", graphId, folderId));
        }
        return new NotFoundException(String.format("no graph with id '%s' in project '%s'", graphId, projectId));
    }

    private static ConflictException graphIsOutOfDate(String projectId, String folderId, String graphId, int version) {
        String message = String.format(
            "graph (%s, %s) with version %s is out of date",
            StringUtils.isEmpty(folderId) ? projectId : folderId,
            graphId,
            version
        );
        return new ConflictException(message);
    }
}
