package ru.yandex.solomon.core.conf;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.ProjectMenuDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.ProjectMenu;
import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author Oleg Baryshnikov
 */
@Component
@Import(ConfigDaoContext.class)
@ParametersAreNonnullByDefault
public class ProjectMenuManager {

    private final ProjectsDao projectsDao;
    private final ProjectMenuDao projectMenuDao;

    @Autowired
    public ProjectMenuManager(ProjectsDao projectsDao, ProjectMenuDao projectMenuDao) {
        this.projectsDao = projectsDao;
        this.projectMenuDao = projectMenuDao;
    }

    public CompletableFuture<ProjectMenu> getProjectMenu(String id) {
        return projectMenuDao.findById(id)
            .thenApply(menu -> menu.orElse(ProjectMenu.empty(id)));
    }

    public CompletableFuture<ProjectMenu> saveProjectMenu(ProjectMenu projectMenu) {
        return checkForeignRefs(projectMenu)
            .thenCompose(aVoid -> projectMenuDao.upsert(projectMenu))
            .thenApply(menuO -> {
                if (menuO.isPresent()) {
                    return menuO.get();
                }
                throw new BadRequestException(String.format("failed to save menu for project %s", projectMenu.getId()));
            });
    }

    private CompletableFuture<Void> checkForeignRefs(ProjectMenu projectMenu) {
        return projectsDao.exists(projectMenu.getId())
            .thenAccept(exists -> {
                if (!exists) {
                    throw new BadRequestException(String.format("project %s does not exist", projectMenu.getId()));
                }
            });
    }
}
