package ru.yandex.solomon.core.conf;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.ProjectSettingsDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.ProjectSettings;
import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author Oleg Baryshnikov
 */
@Component
@Import(ConfigDaoContext.class)
@ParametersAreNonnullByDefault
public class ProjectSettingsManager {
    private final ProjectsDao projectsDao;
    private final ProjectSettingsDao projectSettingsDao;

    @Autowired
    public ProjectSettingsManager(ProjectsDao projectsDao, ProjectSettingsDao projectSettingsDao) {
        this.projectsDao = projectsDao;
        this.projectSettingsDao = projectSettingsDao;
    }

    public CompletableFuture<ProjectSettings> getProjectSettings(String id) {
        return checkForeignRefs(id)
                .thenCompose(aVoid -> projectSettingsDao.find(id));
    }

    public CompletableFuture<ProjectSettings> updateProjectSettings(String id, Map<String, String> params) {
        return checkForeignRefs(id)
                .thenCompose(aVoid -> projectSettingsDao.upsert(new ProjectSettings(id, params)));
    }

    public CompletableFuture<String> getProjectSettingByKey(String id, String key) {
        return checkForeignRefs(id)
                .thenCompose(aVoid -> projectSettingsDao.find(id))
                .thenApply(settings -> settings.getSettings().get(key));
    }

    public CompletableFuture<String> updateProjectSettingByKey(String id, String key, String value) {
        return checkForeignRefs(id)
                .thenCompose(aVoid -> projectSettingsDao.find(id))
                .thenCompose(projectSettings -> {
                    HashMap<String, String> newSettings = new HashMap<>(projectSettings.getSettings());
                    newSettings.put(key, value);
                    ProjectSettings newProjectSettings = new ProjectSettings(id, newSettings);
                    return projectSettingsDao.upsert(newProjectSettings)
                            .thenApply(updatedProjectSettings -> updatedProjectSettings.getSettings().get(key));
                });
    }

    public CompletableFuture<Void> deleteProjectSettingByKey(String id, String key) {
        return checkForeignRefs(id)
                .thenCompose(aVoid -> projectSettingsDao.find(id))
                .thenCompose(projectSettings -> {
                    ProjectSettings newProjectSettings = removeSettingByKey(key, projectSettings);
                    return projectSettingsDao.upsert(newProjectSettings);
                })
                .thenApply(projectSettings -> null);
    }

    private static ProjectSettings removeSettingByKey(String key, ProjectSettings projectSettings) {
        HashMap<String, String> newSettings = new HashMap<>(projectSettings.getSettings());
        newSettings.remove(key);
        return new ProjectSettings(projectSettings.getId(), newSettings);
    }

    private CompletableFuture<Void> checkForeignRefs(String projectId) {
        return projectsDao.exists(projectId)
                .thenAccept(exists -> {
                    if (!exists) {
                        throw new BadRequestException(String.format("project %s does not exist", projectId));
                    }
                });
    }
}
