package ru.yandex.solomon.core.conf;

import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.DashboardsDao;
import ru.yandex.solomon.core.db.dao.GraphsDao;
import ru.yandex.solomon.core.db.dao.ProjectMenuDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServicesDao;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.ProjectPermission;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.core.exceptions.NotOwnerException;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

/**
 * @author Sergey Polovko
 */
@Component
@Import({
    ConfigDaoContext.class,
    ShardsManager.class,
})
public class ProjectsManager {

    private final ProjectsDao projectsDao;
    private final ShardsManager shardsManager;
    private final ServicesDao servicesDao;
    private final ClustersDao clustersDao;
    private final GraphsDao graphsDao;
    private final DashboardsDao dashboardsDao;
    private final ProjectMenuDao projectMenuDao;

    @Autowired
    public ProjectsManager(
            ProjectsDao projectsDao,
            ShardsManager shardsManager,
            ServicesDao servicesDao,
            ClustersDao clustersDao,
            GraphsDao graphsDao,
            DashboardsDao dashboardsDao,
            ProjectMenuDao projectMenuDao)
    {
        this.projectsDao = projectsDao;
        this.shardsManager = shardsManager;
        this.servicesDao = servicesDao;
        this.clustersDao = clustersDao;
        this.graphsDao = graphsDao;
        this.dashboardsDao = dashboardsDao;
        this.projectMenuDao = projectMenuDao;
    }

    public CompletableFuture<List<Project>> getProjects() {
        return projectsDao.findAllNames();
    }

    @Deprecated
    public CompletableFuture<PagedResult<Project>> find(
        String text,
        String abcFilter,
        String login,
        @Nullable EnumSet<ProjectPermission> filterByPermissions,
        PageOptions pageOpts)
    {
        return projectsDao.find(text, abcFilter, login, filterByPermissions, pageOpts);
    }

    public CompletableFuture<PagedResult<Project>> find(
            String text,
            String abcFilter,
            Set<String> projectIds,
            PageOptions pageOpts)
    {
        return projectsDao.findInProjects(text, abcFilter, projectIds, pageOpts);
    }

    public CompletableFuture<TokenBasePage<Project>> findV3(
        String text,
        int pageSize,
        String pageToken)
    {
        return projectsDao.findV3(text, pageSize, pageToken);
    }

    public CompletableFuture<Project> getProject(String id) {
        return projectsDao.findById(id)
            .thenApply(project -> project.orElseThrow(() -> projectNotFound(id)));
    }

    public CompletableFuture<Project> createProject(Project project) {
        return projectsDao.insert(project)
            .thenApply(inserted -> {
                if (!inserted) {
                    throw new ConflictException(String.format("project with id %s already exists", project.getId()));
                }
                return project;
            });
    }

    public CompletableFuture<Void> upsertProjects(List<Project> projects) {
        return projectsDao.upsertProjects(projects);
    }

    public CompletableFuture<Project> updateProject(
            String accountId,
            Project project,
            boolean canUpdateAny,
            boolean canUpdateOldFields)
    {
        return projectsDao.findById(project.getId())
            .thenCompose(project0 -> {
                if (project0.isEmpty()) {
                    throw projectNotFound(project.getId());
                }
                boolean canChangeOwner = canUpdateAny || accountId.equals(project0.get().getOwner());
                return projectsDao.partialUpdate(project, canChangeOwner, canUpdateAny, canUpdateOldFields)
                    .thenApply(updatedProject -> {
                        if (updatedProject.isPresent()) {
                            return updatedProject.get();
                        }
                        throw projectIsOutOfDate(project.getId(), project.getVersion());
                    });
            });
    }

    public CompletableFuture<Void> deleteProject(String accountId, String id, boolean canDeleteByNonOwner, boolean skipValidation) {
        return deleteProjectEntities(accountId, id, canDeleteByNonOwner)
            // (3) delete project itself
            .thenCompose(aVoid -> projectsDao.deleteOne(id, skipValidation))
            .thenAccept(deleted -> {});
    }

    public CompletableFuture<Void> deleteProjectEntities(String accountId, String id, boolean canDeleteByNonOwner) {
        return projectsDao.findById(id)
                .thenAccept(projectO -> {
                    if (projectO.isEmpty()) {
                        throw projectNotFound(id);
                    }

                    String owner = projectO.get().getOwner();
                    if (!canDeleteByNonOwner && !accountId.equals(owner)) {
                        throw new NotOwnerException("you have no permissions to delete project", owner);
                    }
                })
                // (1) delete shards
                .thenCompose(aVoid -> shardsManager.deleteByProjectId(id, ""))
                // (2) delete rest stuff in parallel
                .thenCompose(unit -> {
                    return CompletableFuture.allOf(
                            clustersDao.deleteByProjectId(id, ""),
                            servicesDao.deleteByProjectId(id, ""),
                            graphsDao.deleteByProjectId(id, ""),
                            dashboardsDao.deleteByProjectId(id, ""),
                            projectMenuDao.deleteById(id));
                });
    }

    private static NotFoundException projectNotFound(String id) {
        return new NotFoundException(String.format("no project with id '%s'", id));
    }

    private static ConflictException projectIsOutOfDate(String projectId, int version) {
        String message = String.format(
            "project %s with version %s is out of date",
            projectId,
            version
        );
        return new ConflictException(message);
    }

    public CompletableFuture<TokenBasePage<Project>> listProjects(int pageSize, String token) {
        return projectsDao.list(new TokenPageOptions(pageSize, token));
    }
}
