package ru.yandex.solomon.core.conf;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;

import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.util.ServicePathEncoder;

/**
 * @author checat
 */
@ParametersAreNonnullByDefault
public class ServiceConfDetailed {
    private final String name;
    private final String id;
    private final String path;
    private final Service raw;
    @Nullable
    private final ServiceProvider serviceProvider;

    public ServiceConfDetailed(Service service, @Nullable ServiceProvider serviceProvider) {
        this.id = service.getId();
        this.name = service.getName();
        this.serviceProvider = serviceProvider;
        this.raw = service;
        this.path = ShardSettings.getPath(raw.getShardSettings(), "");

        if (serviceProvider == null && !Strings.isNullOrEmpty(service.getServiceProvider())) {
            throw new IllegalArgumentException("Not found service provider: " + service.getServiceProvider());
        }
        try {
            ServicePathEncoder.encode(this.path);
        } catch (Exception e) {
            throw new IllegalStateException("failed to decode path: \"" + this.path + '"');
        }
    }

    public int getInterval() {
        if (serviceProvider != null && serviceProvider.getShardSettings().getInterval() > 0) {
            return serviceProvider.getShardSettings().getInterval();
        }

        return ShardSettings.getInterval(raw.getShardSettings(), 0);
    }

    public String getName() {
        return name;
    }

    public String getId() {
        return id;
    }

    public String getPath() {
        return path;
    }

    public int getPort() {
        return ShardSettings.getPort(raw.getShardSettings(), 0);
    }

    public Service getRaw() {
        return raw;
    }

    public int getGridSec() {
        if (serviceProvider != null && serviceProvider.getShardSettings().getGrid() > 0) {
            return serviceProvider.getShardSettings().getGrid();
        }
        return ShardSettings.getGrid(raw.getShardSettings(), 0);
    }

    public int getMetricsTtlDays() {
        int ttlDays = 0;
        if (serviceProvider != null) {
            ttlDays = serviceProvider.getShardSettings().getMetricsTtlDays();
        }

        int settingsValue = ShardSettings.getMetricsTtlDays(raw.getShardSettings(), 0);
        if (ttlDays == 0) {
            return settingsValue;
        }

        int minTtlDays = Math.min(ttlDays, settingsValue);
        if (minTtlDays > 0) {
            return minTtlDays;
        }

        return ttlDays;
    }

    public PushOrPull getPushOrPull() {
        ShardSettings.PullSettings pullSettings = raw.getShardSettings().getPullSettings();
        var path = pullSettings == null ? "" : pullSettings.getPath();
        return ShardSettings.getPushOrPull(raw.getShardSettings(),
                !path.isEmpty()
                        ? PushOrPull.PULL
                        : PushOrPull.PUSH);
    }

    @Nullable
    public ServiceProvider getServiceProvider() {
        return serviceProvider;
    }

    public ServiceMetricConf getMetricConf() {
        if (serviceProvider != null) {
            return serviceProvider.getShardSettings().getMetricConf();
        }
        return ShardSettings.toServiceMetricConf(raw.getShardSettings(), raw.getMetricConf());
    }

    public String getMetricNameLabel() {
        return raw.getMetricNameLabel();
    }

    public DecimPolicy getDecimPolicy() {
        return ShardSettings.getDecimPolicy(raw.getShardSettings(), DecimPolicy.UNDEFINED);
    }
}
