package ru.yandex.solomon.core.conf;

import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServicesDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.ClusterServiceAssociation;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;


/**
 * @author Sergey Polovko
 */
@Component
@Import(ConfigDaoContext.class)
public class ServicesManager {

    private final ServicesDao servicesDao;
    private final ProjectsDao projectsDao;
    private final ShardsDao shardsDao;
    private final ShardsManager shardsManager;

    @Autowired
    public ServicesManager(ServicesDao servicesDao, ProjectsDao projectsDao, ShardsDao shardsDao, ShardsManager shardsManager) {
        this.servicesDao = servicesDao;
        this.projectsDao = projectsDao;
        this.shardsDao = shardsDao;
        this.shardsManager = shardsManager;
    }

    public CompletableFuture<PagedResult<Service>> getAllServices(PageOptions pageOpts, String text) {
        return servicesDao.findAll(pageOpts, text);
    }

    public CompletableFuture<PagedResult<Service>> getServices(
            String projectId,
            String folderId,
            PageOptions pageOpts,
            String text,
            ShardSettings.Type monitoringModel)
    {
        return servicesDao.findByProjectId(projectId, folderId, pageOpts, text, monitoringModel);
    }


    public CompletableFuture<TokenBasePage<Service>> getPagedServices(
            String projectId,
            String folderId,
            int pageSize,
            String pageToken,
            String text) {
        return servicesDao.findByProjectIdPaged(projectId, folderId, pageSize, pageToken, text);
    }

    public CompletableFuture<Service> getService(String projectId, String folderId, String serviceId) {
        return servicesDao.findOne(projectId, folderId, serviceId)
            .thenApply(service -> service.orElseThrow(() -> serviceNotFound(projectId, folderId, serviceId)));
    }

    public CompletableFuture<Service> createService(Service service) {
        return checkForeignRefs(service)
            .thenCompose(aVoid -> servicesDao.insert(service))
            .thenApply(inserted -> {
                if (inserted) {
                    return service;
                }
                throw ConflictException.alreadyExists("service", service.getProjectId(), service.getId());
            });
    }

    public CompletableFuture<Service> updateService(Service service) {
        // TODO: use transactions instead of this messy code
        return checkForeignRefs(service)
            .thenCompose(aVoid -> servicesDao.findOne(service.getProjectId(), service.getFolderId(), service.getId()))
            .thenCompose(oldServiceOpt -> {
                if (oldServiceOpt.isEmpty()) {
                    throw serviceNotFound(service.getProjectId(), service.getFolderId(), service.getId());
                }

                Service oldService = oldServiceOpt.get();
                if (service.getVersion() != oldService.getVersion()) {
                    throw serviceIsOutOfDate(service.getProjectId(), service.getFolderId(), service.getId(), service.getVersion());
                }

                if (Objects.equals(service.getName(), oldService.getName())) {
                    // if names are equal then just update
                    return doServiceUpdate(service);
                }

                CompletableFuture<Service> resultFuture = new CompletableFuture<>();
                shardsDao.patchWithServiceName(service.getProjectId(), service.getId(), service.getName())
                    .thenCompose(aVoid2 -> doServiceUpdate(service))
                    .whenComplete((updatedService, throwable) -> {
                        if (throwable == null) {
                            resultFuture.complete(updatedService);
                            return;
                        }
                        // revert shards patch
                        shardsDao.patchWithServiceName(service.getProjectId(), service.getId(), oldService.getName())
                            .whenComplete((aVoid, ignoredException) -> {
                                resultFuture.completeExceptionally(throwable);
                            });
                    });
                return resultFuture;
            });
    }

    private CompletableFuture<Service> doServiceUpdate(Service service) {
        return servicesDao.partialUpdate(service)
            .thenApply(updatedService -> {
                if (updatedService.isPresent()) {
                    return updatedService.get();
                }
                throw serviceNotFound(service.getProjectId(), service.getFolderId(), service.getId());
            });
    }

    public CompletableFuture<Void> deleteService(String projectId, String folderId, String serviceId) {
        return checkRelatedClusters(projectId, folderId, serviceId)
            .thenCompose(aVoid -> servicesDao.deleteOne(projectId, folderId, serviceId))
            .thenAccept(deleted -> {
                if (!deleted) {
                    throw serviceNotFound(projectId, folderId, serviceId);
                }
            });
    }

    public CompletableFuture<List<ClusterServiceAssociation>> getServiceAssociations(
        String projectId,
        String folderId,
        String serviceId)
    {
        return shardsDao.findByServiceId(projectId, folderId, serviceId)
            .thenApply(shards -> shards.stream()
                .map(ClusterServiceAssociation::createAssociationToCluster)
                .collect(Collectors.toList()));
    }

    public CompletableFuture<List<ShardsManager.ShardExtended>> getServiceShards(
            String projectId,
            String folderId,
            String serviceId,
            EnumSet<ShardState> state)
    {
        return shardsDao.findByServiceId(projectId, folderId, serviceId)
                .thenApply(shards -> shards.stream()
                        .filter(shard -> state == null || state.isEmpty() || state.contains(shard.getState()))
                        .collect(Collectors.toList()))
                .thenCompose(shards -> shardsManager.getExtendedShards(shards, projectId, folderId, true));
    }

    public CompletableFuture<TokenBasePage<ClusterServiceAssociation>> getServiceAssociations(
            String projectId,
            String folderId,
            String serviceId,
            Set<ShardState> states,
            String filter,
            int pageSize,
            String pageToken) {
        return shardsDao.findByServiceIdV3(projectId, folderId, serviceId, states, filter, pageSize, pageToken)
                .thenApply(shards -> shards.map(ClusterServiceAssociation::createAssociationToCluster));
    }

    private CompletableFuture<Void> checkForeignRefs(Service service) {
        return projectsDao.exists(service.getProjectId())
            .thenAccept(exists -> {
                if (!exists) {
                    throw new BadRequestException(String.format("project %s does not exist", service.getProjectId()));
                }
            });
    }

    private CompletableFuture<Void> checkRelatedClusters(String projectId, String folderId, String serviceId) {
        return shardsDao.findByServiceId(projectId, folderId, serviceId)
            .thenAccept(shards -> {
                if (!shards.isEmpty()) {
                    List<String> clusterIds = shards.stream()
                        .map(Shard::getClusterId)
                        .collect(Collectors.toList());
                    String message = "service cannot be deleted because it has associated clusters: " + clusterIds;
                    throw new BadRequestException(message);
                }
            });
    }

    private static NotFoundException serviceNotFound(String projectId, String folderId, String serviceId) {
        if (!StringUtils.isEmpty(folderId)) {
            return new NotFoundException(String.format("no service with id '%s' in folder '%s'", serviceId, folderId));
        }
        return new NotFoundException(String.format("no service with id '%s' in project '%s'", serviceId, projectId));
    }

    private static ConflictException serviceIsOutOfDate(String projectId, String folderId, String serviceId, int version) {
        String message = String.format(
            "service (%s, %s) with version %s is out of date",
            StringUtils.isEmpty(folderId) ? projectId : folderId,
            serviceId,
            version
        );
        return new ConflictException(message);
    }
}
