package ru.yandex.solomon.core.conf;

import java.time.Instant;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.core.db.model.ValidationMode;
import ru.yandex.solomon.labels.shard.ShardKey;

/**
 * @author checat
 */
@ParametersAreNonnullByDefault
public class ShardConfDetailed {

    private final Project project;
    private final ClusterConfDetailed cluster;
    private final ServiceConfDetailed service;
    private final int perUrlQuota;
    private final Shard raw;

    public ShardConfDetailed(
        Shard shard,
        Project project,
        ClusterConfDetailed cluster,
        ServiceConfDetailed service)
    {
        this.raw = shard;
        if (shard.getProjectId().isEmpty()) {
            throw new RuntimeException("project is unspecified");
        }

        this.project = project;
        this.cluster = cluster;
        this.service = service;
        this.perUrlQuota = shard.getMaxMetricsPerUrl();
    }

    public Shard getRaw() {
        return raw;
    }

    public int getPerUrlQuota() {
        return perUrlQuota;
    }

    public String getId() {
        return raw.getId();
    }

    public int getNumId() {
        return raw.getNumId();
    }

    public int getNumPartitions() {
        return raw.getNumPartitions();
    }

    public ShardKeyAndId getShardKeyAndId() {
        return new ShardKeyAndId(shardKey(), getId(), getNumId());
    }

    public Project getProject() {
        return project;
    }

    public ClusterConfDetailed getCluster() {
        return cluster;
    }

    public ServiceConfDetailed getService() {
        return service;
    }

    public String getProjectId() {
        return raw.getProjectId();
    }

    public ShardKey shardKey() {
        return new ShardKey(getProjectId(), cluster.getName(), service.getName());
    }

    public Instant getCreatedAt() {
        return raw.getCreatedAt();
    }

    /**
     * @return TTL in days or {@code 0} if TTL is not configured.
     * <p>
     * Defined priorities:
     * - shard     highest priority
     * - cluster
     * - service   lowest priority
     */
    public int getMetricsTtlDays() {
        var metricsTtlDays = ShardSettings.getMetricsTtlDays(raw.getShardSettings(), 0);
        if (metricsTtlDays > 0) {
            return metricsTtlDays;
        }
        if (cluster.getMetricsTtlDays() > 0) {
            return cluster.getMetricsTtlDays();
        }
        return service.getMetricsTtlDays();
    }

    public int getGridSec() {
        var grid = ShardSettings.getGrid(raw.getShardSettings(), 0);
        if (grid > 0) {
            return grid;
        }
        if (cluster.getGridSec() > 0) {
            return cluster.getGridSec();
        }
        if (service.getGridSec() > 0) {
            return service.getGridSec();
        }

        // use interval values
        grid = ShardSettings.getInterval(raw.getShardSettings(), 0);
        if (grid > 0) {
            return grid;
        }
        if (cluster.getInterval() > 0) {
            return cluster.getInterval();
        }
        if (service.getInterval() > 0) {
            return service.getInterval();
        }
        return Service.DEFAULT_INTERVAL;
    }

    public boolean hasGrid() {
        return getGridSec() != Service.GRID_ABSENT;
    }

    public DecimPolicy getDecimPolicy() {
        var decimPolicy = ShardSettings.getDecimPolicy(raw.getShardSettings(), DecimPolicy.UNDEFINED);
        if (decimPolicy.isDefined()) {
            return decimPolicy;
        }
        if (cluster.getDecimPolicy().isDefined()) {
            return cluster.getDecimPolicy();
        }
        if (service.getDecimPolicy().isDefined()) {
            return service.getDecimPolicy();
        }
        if (project.getId().startsWith("yasm_")) {
            return DecimPolicy.YASM_DEFAULT;
        }
        return DecimPolicy.DEFAULT;
    }

    public ValidationMode getValidationMode() {
        return raw.getValidationMode();
    }

    /**
     * Metric name label defined for shard.
     * Returns service metric label name if shard metric name label is empty.
     * It's necessary to migrate to metric name.
     */
    public String getMetricNameLabel() {
        String shardMetricNameLabel = raw.getMetricNameLabel();
        if (!shardMetricNameLabel.isEmpty()) {
            return shardMetricNameLabel;
        }

        String serviceMetricNameLabel = service.getMetricNameLabel();
        if (!serviceMetricNameLabel.isEmpty()) {
            return serviceMetricNameLabel;
        }

        return project.getMetricNameLabel();
    }

    @Override
    public String toString() {
        return "ShardContext[shard=" + raw.getId() +
                ",project=" + raw.getProjectId() +
                ",cluster=" + raw.getClusterId() +
                ",service=" + raw.getServiceId() + ",...]";
    }

    public int getMaxMemMetrics() {
        return raw.getMaxMemMetrics();
    }

    public int getMaxFileMetrics() {
        return raw.getMaxFileMetrics();
    }

    public int getMaxMetricsPerUrl() {
        return raw.getMaxMetricsPerUrl();
    }

    public int getMaxResponseSizeBytes() {
        return raw.getMaxResponseSizeBytes();
    }

    public int getInterval() {
        var stepSeconds = ShardSettings.getInterval(raw.getShardSettings(), 0);
        if (stepSeconds > 0) {
            return stepSeconds;
        }

        stepSeconds = cluster.getInterval();
        if (stepSeconds > 0) {
            return stepSeconds;
        }

        stepSeconds = service.getInterval();
        if (stepSeconds > 0) {
            return stepSeconds;
        }

        return Service.DEFAULT_INTERVAL;
    }

    public ServiceMetricConf getMetricConf() {
        var conf = ShardSettings.toServiceMetricConf(raw.getShardSettings(), null);
        if (conf != null) {
            return conf;
        }
        if (cluster.getMetricConf() != null) {
            return cluster.getMetricConf();
        }
        return service.getMetricConf();
    }

    public PushOrPull getPushOrPull() {
        var pullOrPush = ShardSettings.getPushOrPull(raw.getShardSettings(), null);
        if (pullOrPush != null) {
            return pullOrPush;
        }
        if (cluster.getPushOrPull() != null) {
            return cluster.getPushOrPull();
        }
        return service.getPushOrPull();
    }
}
