package ru.yandex.solomon.core.conf;

import java.security.SecureRandom;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.util.actors.PingActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardNumIdGeneratorImpl implements ShardNumIdGenerator, AutoCloseable {
    private final ShardsDao dao;
    private final SecureRandom random;
    private final PingActorRunner actor;

    private volatile IntSet allocatedNumIds = new IntOpenHashSet();

    public ShardNumIdGeneratorImpl(ShardsDao dao, Executor executor, ScheduledExecutorService timer, MetricRegistry registry) {
        this.dao = dao;
        this.random = new SecureRandom();
        registry.lazyGaugeInt64("conf_db.shard.allocated_num_id", () -> allocatedNumIds.size());
        this.actor = PingActorRunner.newBuilder()
                .pingInterval(Duration.ofDays(1L))
                .operation("reload allocated numIds")
                .backoffDelay(Duration.ofSeconds(1))
                .timer(timer)
                .executor(executor)
                .onPing(ignore -> reloadAllocated())
                .build();
        actor.forcePing();
    }

    @Override
    public int generateNumId(String shardId) {
        while (true) {
            int numId = random.nextInt();
            if (numId == 0) {
                continue;
            }

            if (allocatedNumIds.contains(numId)) {
                continue;
            }

            return numId;
        }
    }

    private CompletableFuture<?> reloadAllocated() {
        return dao.findAllIdToShardId()
                .thenAccept(mapping -> {
                    allocatedNumIds = new IntOpenHashSet(mapping.keySet());
                });
    }

    @Override
    public void close() {
        actor.close();
    }
}
