package ru.yandex.solomon.core.conf;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.util.concurrent.MoreExecutors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.salmon.fetcher.proto.FetcherApiProto;
import ru.yandex.salmon.fetcher.proto.FetcherApiProto.TargetStatus;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServicesDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.ClusterServiceNames;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.coremon.client.CoremonClientContext;
import ru.yandex.solomon.fetcher.client.FetcherClient;
import ru.yandex.solomon.fetcher.client.FetcherClientContext;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.net.KnownDc;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

/**
 * @author Sergey Polovko
 */
@Component
@Import({
    ConfigDaoContext.class,
    FetcherClientContext.class,
    CoremonClientContext.class,
})
public class ShardsManager {

    private final ShardsDao shardsDao;
    private final ProjectsDao projectsDao;
    private final ClustersDao clustersDao;
    private final ServicesDao servicesDao;
    private final FetcherClient fetcherClient;
    private final CoremonClient coremonClient;
    private final ShardNumIdGenerator shardNumIdGenerator;
    private final ShardRemoveHandler shardRemoveHandler;
    private final SolomonConfHolder confHolder;
    private final ShardMetricsQuotaReader shardMetricsQuotaReader;

    @Autowired
    public ShardsManager(
            ShardsDao shardsDao,
            ProjectsDao projectsDao,
            ClustersDao clustersDao,
            ServicesDao servicesDao,
            FetcherClient fetcherClient,
            CoremonClient coremonClient,
            ShardNumIdGenerator shardNumIdGenerator,
            ShardRemoveHandler shardRemoveHandler,
            SolomonConfHolder confHolder,
            Optional<ShardMetricsQuotaReader> shardMetricsQuotaReader)
    {
        this.shardsDao = shardsDao;
        this.shardNumIdGenerator = shardNumIdGenerator;
        this.projectsDao = projectsDao;
        this.clustersDao = clustersDao;
        this.servicesDao = servicesDao;
        this.fetcherClient = fetcherClient;
        this.coremonClient = coremonClient;
        this.shardRemoveHandler = shardRemoveHandler;
        this.confHolder = confHolder;
        this.shardMetricsQuotaReader = shardMetricsQuotaReader.orElse(new ShardMetricsQuotaReaderNoOp());
    }

    public CompletableFuture<PagedResult<ShardExtended>> getAllShards(PageOptions pageOptions, ShardState state, String text) {
        return shardsDao.findAll(pageOptions, state, text)
                .thenApply(shardPagedResult -> {
                    List<ShardExtended> resultShards = new ArrayList<>(shardPagedResult.getResult().size());
                    for (var shard : shardPagedResult.getResult()) {
                        resultShards.add(ShardExtended.simple(shard));
                    }
                    return PagedResult.of(resultShards, pageOptions, shardPagedResult.getTotalCount());
                });
    }

    public CompletableFuture<PagedResult<ShardExtended>> getProjectShards(
        String projectId,
        String folderId,
        PageOptions pageOpts,
        EnumSet<ShardState> state,
        String text,
        boolean fullModel)
    {
        return shardsDao.findByProjectId(projectId, folderId, pageOpts, state, text)
                .thenCompose(shardPagedResult -> {
                    return getExtendedShards(shardPagedResult.getResult(), projectId, folderId, fullModel)
                            .thenApply(resultShards -> PagedResult.of(resultShards, pageOpts, shardPagedResult.getTotalCount()));
                });
    }

    private CompletableFuture<ShardExtended> getExtendedShard(String projectId, String folderId, Shard shard, boolean fullModel, boolean currentMetrics) {
        if (!fullModel) {
            return CompletableFuture.completedFuture(ShardExtended.simple(shard));
        }
        return findService(projectId, folderId, shard.getServiceId(), shard)
                .thenCompose(service -> findCluster(projectId, folderId, shard.getClusterId(), shard)
                        .thenCompose(cluster -> fetchShardTargetsStatus(projectId, folderId, shard)
                                .thenCompose(status -> {
                                    if (currentMetrics) {
                                        return shardMetricsQuotaReader.readCurrentMetricsValue(shard.getId(), shard.getProjectId())
                                                .thenApply(quota -> new ShardExtended(shard, service.orElse(null), cluster.orElse(null), status.targetErrors, status.quotaErrors, quota));
                                    } else {
                                        return CompletableFuture.completedFuture(new ShardExtended(shard, service.orElse(null), cluster.orElse(null), status.targetErrors, status.quotaErrors, ShardMetricsQuotaReader.CurrentQuota.EMPTY));
                                    }
                                })));
    }

    private CompletableFuture<Optional<Service>> findService(String projectId, String folderId, String serviceId, Shard shard) {
        var conf = confHolder.getConf();
        if (conf != null) {
            var shardConf = conf.getShardByNumIdOrNull(shard.getNumId());
            if (shardConf != null && shardConf.conf != null) {
                return CompletableFuture.completedFuture(Optional.of(shardConf.conf.getService().getRaw()));
            }
        }
        return servicesDao.findOne(projectId, folderId, serviceId);
    }

    private CompletableFuture<Optional<Cluster>> findCluster(String projectId, String folderId, String clusterId, Shard shard) {
        var conf = confHolder.getConf();
        if (conf != null) {
            var shardConf = conf.getShardByNumIdOrNull(shard.getNumId());
            if (shardConf != null && shardConf.conf != null) {
                return CompletableFuture.completedFuture(Optional.of(shardConf.conf.getCluster().getRaw()));
            }
        }
        return clustersDao.findOne(projectId, folderId, clusterId);
    }

    private CompletableFuture<ShardStatus> fetchShardTargetsStatus(String projectId, String folderId, Shard shard) {
        List<String> hosts = coremonClient.shardHosts(shard.getNumId());
        AtomicReference<String> quotas = new AtomicReference<>("Ok");
        AtomicReference<Integer> failedTargets = new AtomicReference<>(0);
        CompletableFuture<ShardStatus> result = new CompletableFuture<>();
        CompletableFuture<Void> operator = CompletableFuture.completedFuture(null);
        for (String host : hosts) {
            operator = operator
                    .thenCompose(cluster -> fetcherTargetsStatus(projectId, folderId, shard.getId(), host, null, null, UrlStatusType.QUOTA_ERROR, false, new PageOptions(1, 0))
                            .exceptionally(throwable -> new PagedResult<>(List.of(), 0, 0, 0, 0))
                            .thenCompose(quota -> fetcherTargetsStatus(projectId, folderId, shard.getId(), host, null, null, null, true, new PageOptions(1, 0))
                                    .exceptionally(throwable -> new PagedResult<>(List.of(), 0, 0, 0, 0))
                                    .thenAccept(errors -> {
                                        if (quota.getTotalCount() > 0) {
                                            quotas.set("Error");
                                        }
                                        if (errors.getTotalCount() > failedTargets.get()) {
                                            failedTargets.set(errors.getTotalCount());
                                        }
                                    })));

        }
        operator.whenComplete((unused, throwable) -> {
            if (throwable != null) {
                result.completeExceptionally(throwable);
            } else {
                result.complete(new ShardStatus(quotas.get(), failedTargets.get()));
            }
        });
        return result;
    }

    public CompletableFuture<TokenBasePage<Shard>> getProjectShardsV3(
            String projectId,
            String folderId,
            int pageSize,
            String pageToken,
            String text)
    {
        return shardsDao.findByProjectIdV3(projectId, folderId, pageSize, pageToken, text);
    }

    public CompletableFuture<ShardExtended> getShard(String projectId, String folderId, String shardId, boolean fullModel) {
        return shardsDao.findOne(projectId, folderId, shardId)
                .thenApply(shard -> shard.orElseThrow(() -> shardNotFound(projectId, folderId, shardId)))
                .thenCompose(shard -> getExtendedShard(projectId, folderId, shard, fullModel, fullModel));
    }

    public CompletableFuture<ClusterServiceNames> getClusterServiceNames(
        String projectId,
        String folderId,
        String clusterId,
        String serviceId)
    {
        CompletableFuture<Cluster> clusterFuture = clustersDao
            .findOne(projectId, folderId, clusterId)
            .thenApply(c -> c.orElseThrow(() -> new BadRequestException(String.format("cluster %s does not exist", clusterId))));

        CompletableFuture<Service> serviceFuture = servicesDao
            .findOne(projectId, folderId, serviceId)
            .thenApply(s -> s.orElseThrow(() -> new BadRequestException(String.format("service %s does not exist", serviceId))));

        return clusterFuture.thenCombine(serviceFuture, (cluster, service) -> {
            return new ClusterServiceNames(cluster.getName(), service.getName());
        });
    }

    public CompletableFuture<ShardExtended> createShard(Shard shard, boolean canUpdateInternals, boolean fullModel) {
        return checkShardForeignRefs(shard)
            .thenCompose(aVoid -> {
                final Shard toInsert = patchNewShard(shard, canUpdateInternals);
                return insertShard(toInsert, 0);
            })
            .thenCompose(shardResult -> getExtendedShard(shardResult.getProjectId(), shardResult.getFolderId(), shardResult, fullModel, fullModel));
    }

    private CompletableFuture<Shard> insertShard(Shard shard, int attempt) {
        if (attempt > 10) {
            String message = String.format("cannot insert shard %s after %d attempts", shard.getId(), attempt);
            return CompletableFuture.failedFuture(new RuntimeException(message));
        }

        return shardsDao.insert(shard)
            .thenCompose(inserted -> {
                if (inserted) {
                    // force coremon to assign shard
                    return coremonClient.initShard(shard.getProjectId(), shard.getId(), shard.getNumId())
                        .handle((ignore, e) -> shard);
                }

                return shardsDao.exists(shard.getProjectId(), shard.getFolderId(), shard.getId())
                    .thenCompose(exists -> {
                        if (exists) {
                            throw ConflictException.alreadyExists("shard", shard.getProjectId(), shard.getId());
                        }

                        return shardsDao.findByShardKey(shard.getProjectId(), shard.getClusterName(), shard.getServiceName())
                            .thenCompose(pcsCheck -> {
                                if (pcsCheck.isPresent()) {
                                    String message = String.format(
                                        "shard with cluster name %s and service name %s already exists",
                                        shard.getClusterName(),
                                        shard.getServiceName()
                                    );

                                    throw new ConflictException(message);
                                }

                                // numId already used by another shard, try again
                                var patchedShard = shard.toBuilder()
                                    .setNumId(shardNumIdGenerator.generateNumId(shard.getId()))
                                    .build();
                                return insertShard(patchedShard, attempt + 1);
                            });
                    });
            });
    }

    private Shard patchNewShard(Shard shard, boolean canUpdateInternals) {
        Shard.Builder builder = shard.toBuilder();
        builder.setNumId(shardNumIdGenerator.generateNumId(shard.getId()));

        if (shard.hasAnyNonDefaultQuota() && !canUpdateInternals) {
            builder
                .setMaxFileMetrics(Shard.DEFAULT_FILE_METRIC_QUOTA)
                .setMaxMemMetrics(Shard.DEFAULT_MEM_METRIC_QUOTA)
                .setMaxResponseSizeBytes(Shard.DEFAULT_RESPONSE_SIZE_QUOTA)
                .setMaxMetricsPerUrl(Shard.DEFAULT_PER_URL_QUOTA)
                .setNumPartitions(Shard.DEFAULT_PARTITIONS);
        }

        return builder.build();
    }

    public CompletableFuture<ShardExtended> updateShard(Shard shard, boolean canUpdateInternals, boolean fullModel) {
        return checkShardForeignRefs(shard)
            .thenCompose(aVoid -> {
                return shardsDao.partialUpdate(shard, canUpdateInternals)
                    .thenCompose(updatedShard -> {
                        //noinspection OptionalIsPresent
                        if (updatedShard.isPresent()) {
                            return CompletableFuture.completedFuture(updatedShard.get());
                        }

                        return shardsDao.exists(shard.getProjectId(), shard.getFolderId(), shard.getId())
                            .thenApply(exists -> {
                                if (exists) {
                                    throw shardIsOutOfDate(shard.getProjectId(), shard.getFolderId(), shard.getId(), shard.getVersion());
                                }
                                throw shardNotFound(shard.getProjectId(), shard.getFolderId(), shard.getId());
                            });
                    })
                    .thenCompose(shardResult -> getExtendedShard(shardResult.getProjectId(), shardResult.getFolderId(), shardResult, fullModel, fullModel));
            });
    }

    public CompletableFuture<Void> deleteShard(String projectId, String folderId, String shardId) {
        return shardsDao.findOne(projectId, folderId, shardId)
            .thenCompose(shard -> {
                if (shard.isEmpty()) {
                    throw shardNotFound(projectId, folderId, shardId);
                }

                return deleteShard(shard.get())
                        .thenAccept(deleted -> {
                            if (!deleted) {
                                throw shardNotFound(projectId, folderId, shardId);
                            }
                        });
            });
    }

    CompletableFuture<Void> deleteByProjectId(String projectId, String folderId) {
        return shardsDao.findByProjectId(projectId, folderId)
            .thenCompose(shards -> {
                var it = shards.iterator();
                AsyncActorBody body = () -> {
                    if (!it.hasNext()) {
                        return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                    }

                    return deleteShard(it.next());
                };

                var runner = new AsyncActorRunner(body, MoreExecutors.directExecutor(), 1);
                return runner.start();
            });
    }

    private CompletableFuture<Boolean> deleteShard(Shard shard) {
        return CompletableFutures.safeCall(() -> shardRemoveHandler.remove(shard))
                .thenCompose(unit -> shardsDao.deleteOne(shard.getProjectId(), shard.getFolderId(), shard.getId()));
    }

    public CompletableFuture<PagedResult<TargetStatus>> fetcherTargetsStatus(
        String projectId,
        String folderId,
        String shardId,
        String fetcherHost,
        @Nullable String hostGlob,
        @Nullable KnownDc dc,
        @Nullable UrlStatusType status,
        boolean notOkStatus,
        PageOptions pageOptions)
    {
        return getShard(projectId, folderId, shardId, false)
            .thenCompose(shardExtended -> {
                List<String> shardHosts = coremonClient.shardHosts(shardExtended.shard.getNumId());

                final String host = computeFetcherHost(fetcherHost, shardHosts);

                FetcherApiProto.TargetsStatusRequest.Builder requestBuilder = TargetsStatusHelper.requestBuilder(
                        hostGlob, dc, status,
                        notOkStatus, shardExtended.shard);

                requestBuilder.setOffset(pageOptions.getOffset());

                if (pageOptions.isLimited()) {
                    requestBuilder.setLimit(pageOptions.getSize());
                }

                return fetcherClient.targetsStatus(host, requestBuilder.build())
                    .thenApply(response -> PagedResult.of(response.getTargetsList(), pageOptions, response.getTotalCount()));
            });
    }

    public CompletableFuture<TokenBasePage<TargetStatus>> fetcherTargetsStatusV3(
        String projectId,
        String folderId,
        String shardId,
        String fetcherHost,
        @Nullable String hostGlob,
        @Nullable KnownDc dc,
        @Nullable UrlStatusType status,
        boolean notOkStatus,
        TokenPageOptions pageOptions)
    {
        return getShard(projectId, folderId, shardId, false)
            .thenCompose(shardExtended -> {
                List<String> shardHosts = coremonClient.shardHosts(shardExtended.shard.getNumId());

                final String host = computeFetcherHost(fetcherHost, shardHosts);

                FetcherApiProto.TargetsStatusRequest.Builder requestBuilder = TargetsStatusHelper.requestBuilder(
                        hostGlob, dc, status,
                        notOkStatus, shardExtended.shard);

                requestBuilder.setOffset(pageOptions.getOffset());
                requestBuilder.setLimit(pageOptions.getSize() + 1);

                return fetcherClient.targetsStatus(host, requestBuilder.build())
                    .thenApply(response -> TokenBasePage.of(response.getTargetsList(), pageOptions));
            });
    }

    private String computeFetcherHost(String fetcherHost, List<String> shardHosts) {
        final String host;
        if ("".equals(fetcherHost)) {
            host = shardHosts.get(0);
        } else if (shardHosts.contains(fetcherHost)) {
            host = fetcherHost;
        } else {
            String message = String.format(
                    "given host '%s' does not belong to a shard hosts: %s",
                    fetcherHost, shardHosts);
            throw new BadRequestException(message);
        }
        return host;
    }

    private CompletableFuture<Void> checkShardForeignRefs(Shard shard) {
        String projectId = shard.getProjectId();
        String folderId = shard.getFolderId();
        String clusterId = shard.getClusterId();
        String serviceId = shard.getServiceId();

        // referenced objects must exists and were not deleted

        CompletableFuture<Void> projectExists = projectsDao.exists(projectId)
            .thenAccept(exists -> {
                if (!exists) {
                    throw new BadRequestException(String.format("project %s does not exist", projectId));
                }
            });

        CompletableFuture<Void> clusterExists = clustersDao.exists(projectId, folderId, clusterId)
            .thenAccept(exists -> {
                if (!exists) {
                    throw new BadRequestException(String.format("cluster %s does not exist", clusterId));
                }
            });

        CompletableFuture<Void> serviceExists = servicesDao.exists(projectId, folderId, serviceId)
            .thenAccept(exists -> {
                if (!exists) {
                    throw new BadRequestException(String.format("service %s does not exist", serviceId));
                }
            });

        return CompletableFuture.allOf(projectExists, clusterExists, serviceExists);
    }

    private static NotFoundException shardNotFound(String projectId, String folderId, String shardId) {
        if (!StringUtils.isEmpty(folderId)) {
            return new NotFoundException(String.format("no shard with id '%s' in folder '%s'", shardId, folderId));
        }
        return new NotFoundException(String.format("no shard with id '%s' in project '%s'", shardId, projectId));
    }

    private ConflictException shardIsOutOfDate(String projectId, String folderId, String shardId, int version) {
        String message = String.format(
            "shard (%s, %s) with version %s is out of date",
            StringUtils.isEmpty(folderId) ? projectId : folderId,
            shardId,
            version
        );
        return new ConflictException(message);
    }

    public CompletableFuture<Void> reloadShard(
        String projectId,
        String folderId,
        String shardId,
        @Nullable String hostFilter,
        @Nullable String serviceFilter)
    {
        return shardsDao.findOne(projectId, folderId, shardId)
            .thenCompose(shard -> {
                if (shard.isEmpty()) {
                    throw new UnknownShardException();
                }

                int shardNumId = shard.get().getNumId();
                List<String> hosts = coremonClient.shardHosts(shardNumId);
                if (hostFilter != null) {
                    hosts = hosts.stream()
                        .filter(h -> h.contains(hostFilter))
                        .collect(Collectors.toList());
                }

                List<CompletableFuture<?>> futures = new ArrayList<>();
                for (String host : hosts) {
                    if (serviceFilter == null || "coremon".equals(serviceFilter)) {
                        futures.add(coremonClient.reloadShard(host, projectId, shardId));
                    }
                    if (serviceFilter == null || "fetcher".equals(serviceFilter)) {
                        futures.add(fetcherClient.reloadShard(host, projectId, shardId, shardNumId));
                    }
                }

                if (futures.isEmpty()) {
                    return CompletableFuture.completedFuture(null);
                }

                return CompletableFutures.allOfVoid(futures);
            });
    }

    public CompletableFuture<List<ShardExtended>> getExtendedShards(List<Shard> shards, String projectId, String folderId, boolean fullModel) {
        CompletableFuture<List<ShardExtended>> result = new CompletableFuture<>();
        var actorBody = new ShardExtendedReader(shards, projectId, folderId, fullModel);
        var runner = new AsyncActorRunner(actorBody, MoreExecutors.directExecutor(), 15);
        runner.start().whenComplete((unused, throwable) -> {
            if (throwable != null) {
                result.completeExceptionally(throwable);
            } else {
                result.complete(new ArrayList<>(actorBody.results.values()));
            }
        });
        return result;
    }

    public interface ShardRemoveHandler {
        CompletableFuture<Void> remove(Shard shard);
    }

    public record ShardExtended(Shard shard, @Nullable Service service, @Nullable Cluster cluster, @Nullable Integer targetErrors, @Nullable String quotas, ShardMetricsQuotaReader.CurrentQuota currentQuota) {

        public static ShardExtended simple(Shard shard) {
            return new ShardExtended(shard, null, null, null, "", ShardMetricsQuotaReader.CurrentQuota.EMPTY);
        }

        public boolean isFull() {
            return service != null && cluster != null;
        }
    }

    public record ShardStatus(String quotaErrors, Integer targetErrors) {
    }

    private class ShardExtendedReader implements AsyncActorBody {
        private final Iterator<Shard> it;
        private final String projectId;
        private final String folderId;
        private final boolean fullModel;

        private final ConcurrentMap<String, ShardExtended> results = new ConcurrentHashMap<>();

        public ShardExtendedReader(List<Shard> shards, String projectId, String folderId, boolean fullModel) {
            it = shards.iterator();
            this.projectId = projectId;
            this.folderId = folderId;
            this.fullModel = fullModel;
        }

        @Override
        public CompletableFuture<?> run() {
            if (!it.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var shard = it.next();
            return getExtendedShard(projectId, folderId, shard, fullModel, false)
                    .thenApply(shardExtended -> {
                        results.put(shard.getId(), shardExtended);
                        return null;
                    });
        }
    }
}
