package ru.yandex.solomon.core.conf;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Strings;
import io.grpc.Status;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.dao.ServicesDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.Shard;

import static java.util.concurrent.CompletableFuture.completedFuture;


/**
 * @author checat
 */
@Component
@Import(ConfigDaoContext.class)
public class SolomonConfManager {

    private final ServiceProvidersDao serviceProvidersDao;
    private final ProjectsDao projectsDao;
    private final ShardsDao shardDao;
    private final ClustersDao clusterDao;
    private final ServicesDao serviceDao;

    @Autowired
    public SolomonConfManager(
            ServiceProvidersDao serviceProvidersDao,
            ProjectsDao projectsDao,
            ShardsDao shardDao,
            ClustersDao clusterDao,
            ServicesDao serviceDao)
    {
        this.serviceProvidersDao = serviceProvidersDao;
        this.projectsDao = projectsDao;
        this.shardDao = shardDao;
        this.clusterDao = clusterDao;
        this.serviceDao = serviceDao;
    }

    public CompletableFuture<List<ServiceProvider>> getServiceProviders() {
        return serviceProvidersDao.findAll();
    }

    public CompletableFuture<List<Project>> getProjects() {
        return projectsDao.findAllNames();
    }

    public CompletableFuture<List<Cluster>> getClusters() {
        return clusterDao.findAll();
    }

    public CompletableFuture<List<Service>> getServices() {
        return serviceDao.findAll();
    }

    public CompletableFuture<List<Shard>> getShards() {
        return shardDao.findAll();
    }

    public CompletableFuture<ShardConfDetailed> loadShardConf(String projectId, String shardId) {
        return shardDao.findOne(projectId, "", shardId)
            .thenCompose(shardO -> {
                if (shardO.isPresent()) {
                    return shardToDetailedConf(shardO.get());
                }
                return CompletableFuture.failedFuture(new UnknownShardException());
            });
    }

    private CompletableFuture<ShardConfDetailed> shardToDetailedConf(Shard shard) {
        var projectFuture = loadProject(shard.getProjectId());
        var clusterFuture = loadCluster(shard.getProjectId(), shard.getClusterId());
        var serviceFuture = loadService(shard.getProjectId(), shard.getServiceId());
        return CompletableFuture.allOf(projectFuture, clusterFuture, serviceFuture)
            .thenApply(serviceProvider -> {
                var project = projectFuture.join();
                var cluster = clusterFuture.join();
                var service = serviceFuture.join();
                return new ShardConfDetailed(shard, project, cluster, service);
            });
    }

    private CompletableFuture<Project> loadProject(String projectId) {
        return projectsDao.findById(projectId)
                .thenApply(project -> {
                    if (project.isEmpty()) {
                        throw notFound("project not found by id " + projectId);
                    }

                    return project.get();
                });
    }

    private CompletableFuture<ClusterConfDetailed> loadCluster(String projectId, String clusterId) {
        return clusterDao.findOne(projectId, "", clusterId)
                .thenApply(cluster -> {
                    if (cluster.isEmpty()) {
                        throw notFound("cluster not found by id " + clusterId + " at project " + projectId);
                    }

                    return new ClusterConfDetailed(cluster.get());
                });
    }

    private CompletableFuture<ServiceConfDetailed> loadService(String projectId, String serviceId) {
        return serviceDao.findOne(projectId, "", serviceId)
                .thenCompose(opt -> {
                    if (opt.isEmpty()) {
                        throw notFound("service not found by id " + serviceId + " at project " + projectId);
                    }

                    var service = opt.get();
                    if (Strings.isNullOrEmpty(service.getServiceProvider())) {
                        return completedFuture(new ServiceConfDetailed(service, null));
                    }


                    return loadServiceProvider(service.getServiceProvider())
                            .thenApply(serviceProvider -> new ServiceConfDetailed(service, serviceProvider));
                });
    }

    private CompletableFuture<ServiceProvider> loadServiceProvider(String serviceProviderId) {
        return serviceProvidersDao.read(serviceProviderId)
                .thenApply(serviceProvider -> {
                    if (serviceProvider.isEmpty()) {
                        throw notFound("service provider not found by id " + serviceProviderId);
                    }

                    return serviceProvider.get();
                });
    }

    private RuntimeException notFound(String message) {
        throw new StatusRuntimeExceptionNoStackTrace(Status.NOT_FOUND.withDescription(message));
    }
}
