package ru.yandex.solomon.core.conf;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Interner;
import com.google.common.collect.Interners;
import com.google.common.collect.Iterables;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.labels.shard.ShardKey;

/**
 * @author checat
 */
@ParametersAreNonnullByDefault
public class SolomonConfWithContext {

    public static final SolomonConfWithContext EMPTY = SolomonConfWithContext.create(SolomonRawConf.EMPTY);
    public static final Interner<Key> KEY_INTERNER = Interners.newWeakInterner();

    private final List<ShardConfMaybeWrong> shards;
    private final Map<Key, ClusterConfMaybeWrong> clusterById;
    private final Map<Key, ServiceConfMaybeWrong> serviceById;
    private final Map<String, ShardConfMaybeWrong> shardById;
    private final Int2ObjectMap<ShardConfMaybeWrong> shardByNumId;
    private final Map<String, Project> projectsById;
    private final Map<String, ServiceProvider> serviceProviderById;

    /**
     * Contains only correct shards
     */
    private final Map<ShardKey, ShardConfDetailed> shardByKey;
    private final SolomonRawConf rawConf;

    private SolomonConfWithContext(
            List<ShardConfMaybeWrong> shards,
            Map<Key, ClusterConfMaybeWrong> clusterById,
            Map<Key, ServiceConfMaybeWrong> serviceById,
            Map<String, ShardConfMaybeWrong> shardById,
            Int2ObjectMap<ShardConfMaybeWrong> shardByNumId,
            Map<String, Project> projectsById,
            Map<String, ServiceProvider> serviceProviderById,
            Map<ShardKey, ShardConfDetailed> shardByKey,
            SolomonRawConf rawConf)
    {
        this.shards = shards;
        this.clusterById = clusterById;
        this.serviceById = serviceById;
        this.shardById = shardById;
        this.shardByNumId = shardByNumId;
        this.projectsById = projectsById;
        this.serviceProviderById = serviceProviderById;
        this.shardByKey = shardByKey;
        this.rawConf = rawConf;
    }

    public static SolomonConfWithContext create(SolomonRawConf rawConf) {
        var projectsById = rawConf.getProjects()
            .stream()
            .collect(Collectors.toUnmodifiableMap(Project::getId, Function.identity()));

        var serviceProviderById = rawConf.getServiceProviders()
            .stream()
            .collect(Collectors.toUnmodifiableMap(ServiceProvider::getId, Function.identity()));

        var clusterById = new HashMap<Key, ClusterConfMaybeWrong>(projectsById.size());
        for (var cluster : rawConf.getClusters()) {
            ClusterConfMaybeWrong conf;
            try {
                validateProjectId(projectsById.keySet(), cluster.getProjectId());
                conf = new ClusterConfMaybeWrong(cluster);
            } catch (IllegalArgumentException e) {
                conf = new ClusterConfMaybeWrong(cluster, e);
            }
            clusterById.put(KEY_INTERNER.intern(new Key(cluster.getProjectId(), cluster.getId())), conf);
        }

        var serviceById = new HashMap<Key, ServiceConfMaybeWrong>(projectsById.size());
        for (var rawService : rawConf.getServices()) {
            var service = createService(rawService, projectsById, serviceProviderById);
            serviceById.put(KEY_INTERNER.intern(new Key(service.getProjectId(), service.getId())), service);
        }

        int shardCnt = rawConf.getShards().size();
        var shards = new ArrayList<ShardConfMaybeWrong>(shardCnt);
        var shardById = new HashMap<String, ShardConfMaybeWrong>(shardCnt);
        var shardByNumId = new Int2ObjectOpenHashMap<ShardConfMaybeWrong>(shardCnt);
        var shardByKey = new HashMap<ShardKey, ShardConfDetailed>(shardCnt);
        var shardsByProjectId = new HashMap<String, List<ShardConfDetailed>>();

        for (var rawShard : rawConf.getShards()) {
            var shard = createShard(rawShard, projectsById, clusterById, serviceById);
            shards.add(shard);
            shardById.put(shard.getId(), shard);
            shardByNumId.put(shard.getNumId(), shard);
            if (shard.isCorrect()) {
                var correctShard = shard.getConfOrThrow();
                shardByKey.put(correctShard.shardKey(), correctShard);
                shardsByProjectId
                    .computeIfAbsent(correctShard.getProjectId(), ignore -> new ArrayList<>())
                    .add(correctShard);
            }
        }

        return new SolomonConfWithContext(
            shards,
            clusterById,
            serviceById,
            shardById,
            shardByNumId,
            projectsById,
            serviceProviderById,
            shardByKey,
            rawConf);
    }

    private static ShardConfMaybeWrong createShard(
        Shard shard,
        Map<String, Project> projectsById,
        Map<Key, ClusterConfMaybeWrong> clusterById,
        Map<Key, ServiceConfMaybeWrong> serviceById)
    {
        try {
            validateProjectId(projectsById.keySet(), shard.getProjectId());

            Project project = projectsById.get(shard.getProjectId());
            ensureFound(project, "project", shard.getProjectId());

            ClusterConfMaybeWrong cluster = clusterById.get(new Key(shard.getProjectId(), shard.getClusterId()));
            ensureFound(cluster, "cluster", shard.getClusterId());

            ServiceConfMaybeWrong service = serviceById.get(new Key(shard.getProjectId(), shard.getServiceId()));
            ensureFound(service, "service", shard.getServiceId());

            ShardConfDetailed shardConfDetailed = new ShardConfDetailed(
                shard,
                project,
                cluster.getConfOrThrow(),
                service.getConfOrThrow());

            return new ShardConfMaybeWrong(shard, shardConfDetailed);
        } catch (Exception e) {
            return new ShardConfMaybeWrong(shard, e);
        }
    }

    private static <T> void ensureFound(@Nullable T entry, String type, String id) {
        if (entry == null) {
            throw new ValidationException(type + " not found by id: " + id);
        }
    }

    private static ServiceConfMaybeWrong createService(Service service, Map<String, Project> projectsById, Map<String, ServiceProvider> serviceProviderById) {
        try {
            validateProjectId(projectsById.keySet(), service.getProjectId());
            return new ServiceConfMaybeWrong(service, serviceProviderById.get(service.getServiceProvider()));
        } catch (IllegalArgumentException e) {
            return new ServiceConfMaybeWrong(service, e);
        }
    }

    private static void validateProjectId(Set<String> projectIds, String projectId) throws IllegalArgumentException {
        if (!projectIds.contains(projectId)) {
            throw new ValidationException("unknown project: " + projectId);
        }
    }

    public SolomonRawConf getRawConf() {
        return rawConf;
    }

    public List<ShardConfMaybeWrong> getShards() {
        return shards;
    }

    public Iterable<Shard> getAllRawShards() {
        return Iterables.concat(rawConf.getShards(), rawConf.getInactiveShards());
    }

    public int getAllShardsCount() {
        return rawConf.getShards().size() + rawConf.getInactiveShards().size();
    }

    public List<ShardConfDetailed> getCorrectShards() {
        return getCorrectShardsStream().collect(Collectors.toList());
    }

    public Stream<ShardConfDetailed> getCorrectShardsStream() {
        return shardByKey.values().stream();
    }

    public List<ShardKey> getShardKeys() {
        return getCorrectShards().stream().map(ShardConfDetailed::shardKey)
            .collect(Collectors.toList());
    }

    @Nullable
    public ServiceProvider getServiceProvider(String id) {
        return serviceProviderById.get(id);
    }

    @Nullable
    public Project getProject(String projectId) {
        return projectsById.get(projectId);
    }

    public Set<String> projects() {
        return projectsById.keySet();
    }

    @Nullable
    public ClusterConfDetailed getClusterOrNull(String projectId, String clusterId) {
        ClusterConfMaybeWrong clusterConf = clusterById.get(new Key(projectId, clusterId));
        if (clusterConf != null && clusterConf.isCorrect()) {
            return clusterConf.conf;
        }
        return null;
    }

    @Nullable
    public ServiceConfDetailed getServiceOrNull(String projectId, String clusterId) {
        ServiceConfMaybeWrong serviceConf = serviceById.get(new Key(projectId, clusterId));
        if (serviceConf != null && serviceConf.isCorrect()) {
            return serviceConf.conf;
        }
        return null;
    }

    public ShardConfDetailed findShardByKey(ShardKey key) {
        var shard = shardByKey.get(key);
        if (shard == null) {
            throw new UnknownShardException(key);
        }
        return shard;
    }

    public ShardConfMaybeWrong getShardByNumId(int numId) {
        var shard = shardByNumId.get(numId);
        if (shard == null) {
            throw new UnknownShardException(numId);
        }

        return shard;
    }

    @Nullable
    public ShardConfMaybeWrong getShardByIdOrNull(String id) {
        return shardById.get(id);
    }

    @Nullable
    public ShardConfMaybeWrong getShardByNumIdOrNull(int numId) {
        return shardByNumId.get(numId);
    }

    @Nullable
    public ShardConfDetailed findShardOrNull(ShardKey key) {
        return shardByKey.get(key);
    }

    public SolomonConfWithContext addShard(ShardConfDetailed shardConf) {
        Project project = shardConf.getProject();
        Shard shard = shardConf.getRaw();
        Cluster cluster = shardConf.getCluster().getRaw();
        Service service = shardConf.getService().getRaw();

        return create(rawConf.addShard(project, shard, cluster, service));
    }

    private static class ValidationException extends IllegalArgumentException {
        public ValidationException(String s) {
            super(s);
        }

        @Override
        public synchronized Throwable fillInStackTrace() {
            return this;
        }
    }

    private record Key(String projectId, String anotherId) {}
}
