package ru.yandex.solomon.core.conf;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardState;

/**
 * @author Stepan Koltsov
 */
public class SolomonRawConf {

    public static final SolomonRawConf EMPTY = new SolomonRawConf(List.of(), List.of(), List.of(), List.of(), List.of());

    private final List<ServiceProvider> serviceProviders;
    private final List<Project> projects;
    private final List<Cluster> clusters;
    private final List<Service> services;
    private final List<Shard> shards;
    private final List<Shard> inactiveShards;

    public SolomonRawConf(
        List<ServiceProvider> serviceProviders,
        List<Project> projects,
        List<Cluster> clusters,
        List<Service> services,
        List<Shard> allShards)
    {
        this.serviceProviders = serviceProviders;
        this.projects = projects;
        this.clusters = clusters;
        this.services = services;
        this.shards = allShards.stream()
            .filter(s -> s.getState() != ShardState.INACTIVE)
            .collect(Collectors.toList());
        this.inactiveShards = allShards.stream()
            .filter(s -> s.getState() == ShardState.INACTIVE)
            .collect(Collectors.toList());

        // check unique
        checkUnique(projects, Project::getId);
        checkUnique(allShards, Shard::getNumId);
    }

    @SuppressWarnings("ResultOfMethodCallIgnored")
    private static <T, V> void checkUnique(List<T> confs, Function<T, V> idMapper) {
        confs.stream().collect(Collectors.toMap(idMapper, Function.identity()));
    }

    public List<ServiceProvider> getServiceProviders() {
        return serviceProviders;
    }

    public List<String> getProjectIds() {
        return projects.stream().map(Project::getId).collect(Collectors.toList());
    }

    public List<Project> getProjects() {
        return projects;
    }

    public List<Cluster> getClusters() {
        return clusters;
    }

    public List<Service> getServices() {
        return services;
    }

    public List<Shard> getShards() {
        return shards;
    }

    public List<Shard> getInactiveShards() {
        return inactiveShards;
    }

    public boolean isEmpty() {
        return projects.isEmpty() && clusters.isEmpty() && services.isEmpty() && shards.isEmpty();
    }

    private boolean hasCluster(Cluster cluster) {
        for (var c : clusters) {
            if (c.getProjectId().equals(cluster.getProjectId()) && cluster.getId().equals(c.getId())) {
                return true;
            }
        }

        return false;
    }

    private boolean hasService(Service service) {
        for (var s : services) {
            if (s.getProjectId().equals(service.getProjectId()) && service.getId().equals(s.getId())) {
                return true;
            }
        }

        return false;
    }

    private boolean hasProject(Project project) {
        for (var p : projects) {
            if (project.getId().equals(p.getId())) {
                return true;
            }
        }
        return false;
    }

    SolomonRawConf addShard(Project project, Shard shard, Cluster cluster, Service service) {
        List<Shard> newShards = new ArrayList<>(shards);
        newShards.removeIf(s -> {
            // remove shard with the same numId
            if (s.getNumId() == shard.getNumId()) {
                return true;
            }
            // or with the same (project, cluster, service)
            return s.getProjectId().equals(shard.getProjectId()) &&
                s.getClusterName().equals(shard.getClusterName()) &&
                s.getServiceName().equals(shard.getServiceName());
        });
        newShards.add(shard);

        List<Cluster> newClusters = clusters;
        if (!hasCluster(cluster)) {
            newClusters = new ArrayList<>(clusters);
            newClusters.add(cluster);
        }

        List<Service> newServices = services;
        if (!hasService(service)) {
            newServices = new ArrayList<>(services);
            newServices.add(service);
        }

        List<Project> newProjects = projects;
        if (!hasProject(project)) {
            newProjects = new ArrayList<>(projects);
            newProjects.add(project);
        }

        return new SolomonRawConf(serviceProviders, newProjects, newClusters, newServices, newShards);
    }
}
