package ru.yandex.solomon.core.conf;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.UserSettingsDao;
import ru.yandex.solomon.core.db.model.UserSettings;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;

/**
 * @author Oleg Baryshnikov
 */
@Component
@Import(ConfigDaoContext.class)
@ParametersAreNonnullByDefault
public class UserSettingsManager {

    private final UserSettingsDao userSettingsDao;

    @Autowired
    public UserSettingsManager(UserSettingsDao userSettingsDao) {
        this.userSettingsDao = userSettingsDao;
    }

    public CompletableFuture<UserSettings> getUserSettings(String login) {
        return userSettingsDao.findByLogin(login);
    }

    public CompletableFuture<UserSettings> updateUserSettings(String login, Map<String, String> params) {
        return userSettingsDao.upsert(new UserSettings(login, params));
    }

    public CompletableFuture<String> getUserSettingByKey(String login, String key) {
        return userSettingsDao.findByLogin(login).thenApply(settings -> settings.getSettings().get(key));
    }

    public CompletableFuture<String> updateUserSettingByKey(String login, String key, String value) {
        return userSettingsDao.findByLogin(login).thenCompose(userSettings -> {
            HashMap<String, String> newSettings = new HashMap<>(userSettings.getSettings());
            newSettings.put(key, value);
            UserSettings newUserSettings = new UserSettings(login, newSettings);
            return userSettingsDao.upsert(newUserSettings)
                .thenApply(updatedUserSettings -> updatedUserSettings.getSettings().get(key));
        });
    }

    public CompletableFuture<Void> deleteUserSettingByKey(String login, String key) {
        return userSettingsDao.findByLogin(login).thenCompose(userSettings -> {
            UserSettings newUserSettings = removeSettingByKey(key, userSettings);
            return userSettingsDao.upsert(newUserSettings).thenAccept(result -> {});
        });
    }

    @SuppressWarnings("unused")
    @ManagerMethod
    public CompletableFuture<List<UserSettings>> getAllUserSettings() {
        return userSettingsDao.findAll();
    }

    @SuppressWarnings("unused")
    @ManagerMethod
    public CompletableFuture<Void> deleteUserSettingByKey(String key) {
        return userSettingsDao.findAll().thenCompose(userSettingsList -> {
            List<CompletableFuture<Void>> futures = userSettingsList.stream()
                .map(setting -> removeSettingByKey(key, setting))
                .map(this::deleteOrUpsertSettings)
                .collect(Collectors.toList());

            return CompletableFutures.allOf(futures).thenApply(result -> null);
        });
    }

    @SuppressWarnings("unused")
    @ManagerMethod
    public CompletableFuture<List<UserSettings>> getAllUsersWithSettingValue(String key, String expectedValue) {
        return userSettingsDao.findAll().thenApply(userSettingsList ->
            userSettingsList.stream()
                .filter(userSettings -> {
                    String actualValue = userSettings.getSettings().get(key);
                    if (actualValue == null) {
                        return StringUtils.isBlank(expectedValue);
                    }
                    return actualValue.equals(expectedValue);
                }).collect(Collectors.toList()));
    }

    private CompletableFuture<Void> deleteOrUpsertSettings(UserSettings setting) {
        if (setting.getSettings().isEmpty()) {
            return userSettingsDao.deleteUser(setting.getLogin()).thenApply(result -> null);
        }
        return userSettingsDao.upsert(setting).thenApply(result -> null);
    }

    private UserSettings removeSettingByKey(String key, UserSettings userSettings) {
        HashMap<String, String> newSettings = new HashMap<>(userSettings.getSettings());
        newSettings.remove(key);
        return new UserSettings(userSettings.getLogin(), newSettings);
    }
}
