package ru.yandex.solomon.core.conf;

import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.ViewHistoryDao;
import ru.yandex.solomon.core.db.model.ViewHistory;


/**
 * @author albazh
 */
@Component
@Import(ConfigDaoContext.class)
@ParametersAreNonnullByDefault
public class ViewHistoryManager {

    @Autowired
    private ViewHistoryDao viewHistoryDao;

    public CompletableFuture<Optional<ViewHistory>> updateHistory(
        ViewHistory viewHistory,
        Predicate<ViewHistory> matcher)
    {
        CompletableFuture<String> firstIdFuture;
        if (StringUtils.isEmpty(viewHistory.getId())) {
            firstIdFuture = findFirst(viewHistory.getLogin(), matcher)
                .thenApply(vh -> vh.map(ViewHistory::getId)
                    .orElse(UUID.randomUUID().toString()));
        } else {
            firstIdFuture = CompletableFuture.completedFuture(viewHistory.getId());
        }

        return firstIdFuture.thenCompose(firstId -> viewHistoryDao.upsert(viewHistory.withId(firstId)));
    }

    public CompletableFuture<Boolean> deleteByLoginAndId(String login, String id) {
        return viewHistoryDao.deleteById(login, id);
    }

    public CompletableFuture<Optional<ViewHistory>> findFirst(String login, Predicate<ViewHistory> filter) {
        return viewHistoryDao.findHistoryByLogin(login)
            .thenApply(histories -> histories.stream()
                .filter(filter)
                .findFirst());
    }

    public CompletableFuture<List<ViewHistory>> findAll(String login, Predicate<ViewHistory> filter) {
        return viewHistoryDao.findHistoryByLogin(login)
            .thenApply(histories -> histories.stream()
                .filter(filter)
                .collect(Collectors.toList()));
    }
}
