package ru.yandex.solomon.core.conf.aggr;

import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class AggrRuleConf extends DefaultObject {
    private final LabelCond cond;
    private final List<TargetParsed> target;
    private final MetricAggregation aggregation;

    public AggrRuleConf(ServiceMetricConf.AggrRule aggrRule, ShardKey shardKey) {
        // sanity
        if (aggrRule.getTarget().length == 0) {
            throw new RuntimeException("Empty target conf: " + shardKey);
        }

        this.cond = LabelCond.parse(aggrRule.getCond());
        this.target = Arrays.stream(aggrRule.getTarget())
                .flatMap(t -> {
                    TargetParsed r = TargetParsed.parseReplaceSpecials(t, shardKey, cond.keysOfAbsentSelectors());
                    return r != null ? Stream.of(r) : Stream.empty();
                })
                .filter(t -> !t.getName().equals("summary")) // TODO: before deleting this filter make sure that Samogon
                .collect(Collectors.toList());               //       does not create services with such aggregation rules.
        this.aggregation = Nullables.orDefault(aggrRule.getAggr(), MetricAggregation.SUM);
    }

    public static AggrRuleConf simpleReplaceRule(@Nonnull String label, @Nonnull String replaceValue) {
        ServiceMetricConf.AggrRule aggrRule = new ServiceMetricConf.AggrRule(
                new String[]{label + "=*"},
                new String[]{label + "=" + replaceValue},
                null);
        return new AggrRuleConf(
            aggrRule,
            new ShardKey("fake-project", "fake-cluster", "fake-service")
        );
    }

    public Set<String> getConditionLabelNames() {
        return cond.getConditionLabelNames();
    }

    public Set<String> getSubstitutionLabelNames() {
        return target.stream()
                .flatMap(t -> t.getSubstitutionLabelNames().stream())
                .collect(Collectors.toSet());
    }

    public MetricAggregation getAggregation() {
        return aggregation;
    }

    public boolean isHostRequired() {
        if (cond.isHostRequired()) {
            return true;
        }

        boolean replaceHost = false;
        for (var target : target) {
            if (target.isHostRequired()) {
                return true;
            }

            replaceHost |= LabelKeys.HOST.equals(target.getName());
        }

        return !replaceHost;
    }

    @Nullable
    public Labels matchTransform(Labels labels, Labels optLabels, LabelAllocator labelAllocator) {
        if (!this.cond.allMatch(labels, optLabels)) {
            return null;
        }

        var builder = Labels.builder(Labels.MAX_LABELS_COUNT, labelAllocator);
        builder.addAll(labels);

        for (TargetParsed targetItem : target) {
            if (!targetItem.apply(builder, labels, optLabels)) {
                return null;
            }
        }

        Labels r = builder.build();
        if (r.equals(labels)) {
            return null;
        }

        return r;
    }
}
