package ru.yandex.solomon.core.conf.aggr;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import com.google.common.collect.ImmutableSet;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.selector.LabelSelector;
import ru.yandex.solomon.labels.selector.LabelSelector.MatchOp;
import ru.yandex.solomon.labels.selector.LabelSelectorParsed;


/**
 * @author Stepan Koltsov
 */
public class LabelCond extends DefaultObject {

    private final CondAtom[] atoms;

    private LabelCond(CondAtom[] atoms) {
        this.atoms = atoms;
    }

    public boolean allMatch(Labels labels) {
        for (CondAtom cond : atoms) {
            if (!cond.match(labels)) {
                return false;
            }
        }
        return true;
    }

    public boolean allMatch(Labels labels, Labels optLabels) {
        for (CondAtom cond : atoms) {
            if (!cond.match(labels, optLabels)) {
                return false;
            }
        }
        return true;
    }

    public boolean isHostRequired() {
        for (CondAtom cond : atoms) {
            if (LabelKeys.HOST.equals(cond.parsed.getName())) {
                if (cond.parsed.getMatchOp() != MatchOp.PRESENT) {
                    return true;
                }
            }
        }
        return false;
    }

    @SuppressWarnings("CheckReturnValue")
    public Set<String> keysOfAbsentSelectors() {
        ImmutableSet.Builder<String> builder = ImmutableSet.builder();
        for (CondAtom atom : atoms) {
            if (atom.parsed.getMatchOp() == LabelSelector.MatchOp.ABSENT) {
                builder.add(atom.parsed.getName());
            }
        }
        return builder.build();
    }

    public Set<String> getConditionLabelNames() {
        var result = new HashSet<String>(atoms.length);
        for (var atom : atoms) {
            result.add(atom.parsed.getName());
        }
        return result;
    }

    public static LabelCond parse(String[] condList) {
        return new LabelCond(Arrays.stream(condList)
                .map(CondAtom::parse)
                .toArray(CondAtom[]::new));
    }

    private static class CondAtom extends DefaultObject {
        private final LabelSelectorParsed parsed;

        private CondAtom(LabelSelector labelSelector) {
            parsed = labelSelector.toParsed();
        }

        public static CondAtom parse(String string) {
            return new CondAtom(LabelSelector.parse(string));
        }

        public boolean match(Labels labels) {
            return parsed.matches(labels);
        }

        public boolean match(Labels labels, Labels optLabels) {
            Label label = labels.findByKey(parsed.getName());
            final String value;
            if (label == null) {
                Label optLabel = optLabels.findByKey(parsed.getName());
                value = (optLabel == null ? null : optLabel.getValue());
            } else {
                value = label.getValue();
            }
            return parsed.matchValue(value);
        }
    }
}
